#!/bin/bash

# Script de déploiement pour la production (SANS Docker)
# Usage: ./deploy-prod.sh

set -e  # Arrêter en cas d'erreur

echo "🚀 Début du déploiement..."

# Couleurs pour les messages
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Vérifier si on est dans le bon répertoire
if [ ! -f "composer.json" ]; then
    echo -e "${RED}❌ Ce script doit être exécuté depuis le répertoire racine du projet.${NC}"
    exit 1
fi

# Vérifier si Composer est installé
if ! command -v composer &> /dev/null; then
    echo -e "${RED}❌ Composer n'est pas installé. Veuillez l'installer d'abord.${NC}"
    exit 1
fi

# Vérifier si PHP est installé
if ! command -v php &> /dev/null; then
    echo -e "${RED}❌ PHP n'est pas installé. Veuillez l'installer d'abord.${NC}"
    exit 1
fi

# Vérifier si le fichier .env existe
if [ ! -f .env ]; then
    echo -e "${YELLOW}⚠️  Le fichier .env n'existe pas.${NC}"
    if [ -f ENV_EXAMPLE.txt ]; then
        echo -e "${YELLOW}📋 Copie de ENV_EXAMPLE.txt vers .env...${NC}"
        cp ENV_EXAMPLE.txt .env
        echo -e "${YELLOW}⚠️  Veuillez configurer le fichier .env avant de continuer.${NC}"
        exit 1
    else
        echo -e "${RED}❌ Le fichier ENV_EXAMPLE.txt n'existe pas non plus.${NC}"
        exit 1
    fi
fi

echo -e "${GREEN}✅ Vérifications préliminaires réussies${NC}"

# Récupérer les dernières modifications (si Git est utilisé)
if [ -d .git ]; then
    echo -e "${YELLOW}📥 Récupération des modifications Git...${NC}"
    git pull origin main || git pull origin master || echo -e "${YELLOW}⚠️  Pas de repository Git ou problème avec Git pull${NC}"
fi

# Installer les dépendances
echo -e "${YELLOW}📦 Installation des dépendances...${NC}"
composer install --no-dev --optimize-autoloader --no-interaction

# Créer la base de données si elle n'existe pas
echo -e "${YELLOW}💾 Vérification de la base de données...${NC}"
php bin/console doctrine:database:create --if-not-exists || echo -e "${YELLOW}⚠️  Base de données existe déjà ou erreur${NC}"

# Exécuter les migrations
echo -e "${YELLOW}🔄 Exécution des migrations...${NC}"
php bin/console doctrine:migrations:migrate --no-interaction

# Vider le cache
echo -e "${YELLOW}🧹 Vidage du cache...${NC}"
php bin/console cache:clear --env=prod --no-debug

# Réchauffer le cache
echo -e "${YELLOW}🔥 Réchauffement du cache...${NC}"
php bin/console cache:warmup --env=prod

# Définir les permissions (nécessite sudo)
echo -e "${YELLOW}🔐 Configuration des permissions...${NC}"
if [ "$EUID" -eq 0 ]; then
    chown -R www-data:www-data .
    chmod -R 755 .
    chmod -R 775 var
    chmod -R 775 public/uploads
else
    echo -e "${YELLOW}⚠️  Les permissions nécessitent sudo. Exécutez manuellement:${NC}"
    echo -e "${YELLOW}   sudo chown -R www-data:www-data $(pwd)${NC}"
    echo -e "${YELLOW}   sudo chmod -R 755 $(pwd)${NC}"
    echo -e "${YELLOW}   sudo chmod -R 775 $(pwd)/var${NC}"
    echo -e "${YELLOW}   sudo chmod -R 775 $(pwd)/public/uploads${NC}"
fi

echo -e "${GREEN}✅ Déploiement terminé avec succès!${NC}"

# Vérifier les services
echo -e "${YELLOW}📊 Vérification des services...${NC}"
if command -v systemctl &> /dev/null; then
    systemctl is-active --quiet nginx && echo -e "${GREEN}✅ Nginx est actif${NC}" || echo -e "${YELLOW}⚠️  Nginx n'est pas actif${NC}"
    systemctl is-active --quiet php8.2-fpm && echo -e "${GREEN}✅ PHP-FPM est actif${NC}" || echo -e "${YELLOW}⚠️  PHP-FPM n'est pas actif${NC}"
    systemctl is-active --quiet mysql && echo -e "${GREEN}✅ MySQL est actif${NC}" || echo -e "${YELLOW}⚠️  MySQL n'est pas actif${NC}"
fi

echo -e "${GREEN}🌐 L'application devrait être accessible${NC}"

