<?php

namespace App\Command;

use App\Entity\Categorie;
use App\Entity\SousCategorie;
use App\Entity\Caracteristique;
use App\Entity\Produit;
use App\Entity\Client;
use App\Entity\Fournisseur;
use App\Entity\Magasin;
use App\Entity\Achat;
use App\Entity\LigneAchat;
use App\Entity\Paiement;
use App\Entity\Vente;
use App\Entity\LigneVente;
use App\Entity\PaiementClient;
use App\Entity\User;
use App\Service\FactureService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(
    name: 'app:load-test-data',
    description: 'Charge des données de test réalistes dans la base de données',
)]
class LoadTestDataCommand extends Command
{
    public function __construct(
        private EntityManagerInterface $em,
        private FactureService $factureService
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this->addOption('force', 'f', InputOption::VALUE_NONE, 'Supprimer les données existantes avant de créer de nouvelles données');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->title('Chargement des données de test');

        // Récupérer le premier magasin
        $magasin = $this->em->getRepository(Magasin::class)->findOneBy([]);
        if (!$magasin) {
            $io->error('Aucun magasin trouvé. Veuillez d\'abord créer un magasin.');
            return Command::FAILURE;
        }

        // Option pour supprimer les données existantes
        if ($input->getOption('force')) {
            $io->warning('Suppression des données existantes...');
            $this->deleteExistingData($magasin);
            $io->success('Données existantes supprimées.');
        }

        $io->section('Création des catégories et sous-catégories');

        // 1. TÉLÉPHONES
        $categorieTelephones = $this->createCategorie($magasin, 'Téléphones', 'Smartphones et téléphones portables');
        $sousCatIphone = $this->createSousCategorie($categorieTelephones, $magasin, 'iPhone', 'Smartphones Apple');
        $sousCatSamsung = $this->createSousCategorie($categorieTelephones, $magasin, 'Samsung', 'Smartphones Samsung');
        $sousCatXiaomi = $this->createSousCategorie($categorieTelephones, $magasin, 'Xiaomi', 'Smartphones Xiaomi');
        $sousCatHuawei = $this->createSousCategorie($categorieTelephones, $magasin, 'Huawei', 'Smartphones Huawei');
        $sousCatAutres = $this->createSousCategorie($categorieTelephones, $magasin, 'Autres marques', 'Autres marques de smartphones');

        // Caractéristiques pour téléphones
        $this->createCaracteristique($sousCatIphone, $magasin, 'Modèle', Caracteristique::TYPE_SELECT, ['iPhone 15 Pro Max', 'iPhone 15 Pro', 'iPhone 15', 'iPhone 14 Pro Max', 'iPhone 14 Pro', 'iPhone 14', 'iPhone 13 Pro Max', 'iPhone 13 Pro', 'iPhone 13', 'iPhone 12', 'iPhone 11', 'iPhone SE'], true);
        $this->createCaracteristique($sousCatIphone, $magasin, 'Capacité de stockage', Caracteristique::TYPE_SELECT, ['128 GB', '256 GB', '512 GB', '1 TB'], true);
        $this->createCaracteristique($sousCatIphone, $magasin, 'Couleur', Caracteristique::TYPE_SELECT, ['Noir', 'Blanc', 'Bleu', 'Rose', 'Violet', 'Rouge', 'Or'], false);
        $this->createCaracteristique($sousCatIphone, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon', 'Occasion - Moyen'], true);

        $this->createCaracteristique($sousCatSamsung, $magasin, 'Modèle', Caracteristique::TYPE_SELECT, ['Galaxy S24 Ultra', 'Galaxy S24+', 'Galaxy S24', 'Galaxy S23 Ultra', 'Galaxy S23+', 'Galaxy S23', 'Galaxy A54', 'Galaxy A34', 'Galaxy Note 20', 'Galaxy Z Fold 5', 'Galaxy Z Flip 5'], true);
        $this->createCaracteristique($sousCatSamsung, $magasin, 'Capacité de stockage', Caracteristique::TYPE_SELECT, ['128 GB', '256 GB', '512 GB', '1 TB'], true);
        $this->createCaracteristique($sousCatSamsung, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['8 GB', '12 GB', '16 GB'], false);
        $this->createCaracteristique($sousCatSamsung, $magasin, 'Couleur', Caracteristique::TYPE_SELECT, ['Noir', 'Blanc', 'Violet', 'Bleu', 'Vert', 'Rose'], false);
        $this->createCaracteristique($sousCatSamsung, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon', 'Occasion - Moyen'], true);

        $this->createCaracteristique($sousCatXiaomi, $magasin, 'Modèle', Caracteristique::TYPE_SELECT, ['Redmi Note 13 Pro', 'Redmi Note 13', 'Redmi 13', 'POCO X6 Pro', 'POCO F6', 'Mi 14', 'Mi 13'], true);
        $this->createCaracteristique($sousCatXiaomi, $magasin, 'Capacité de stockage', Caracteristique::TYPE_SELECT, ['128 GB', '256 GB', '512 GB'], true);
        $this->createCaracteristique($sousCatXiaomi, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['6 GB', '8 GB', '12 GB', '16 GB'], false);
        $this->createCaracteristique($sousCatXiaomi, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon'], true);

        // 2. ORDINATEURS PORTABLES
        $categorieLaptops = $this->createCategorie($magasin, 'Ordinateurs Portables', 'Laptops et notebooks');
        $sousCatHP = $this->createSousCategorie($categorieLaptops, $magasin, 'HP', 'Ordinateurs HP');
        $sousCatDell = $this->createSousCategorie($categorieLaptops, $magasin, 'Dell', 'Ordinateurs Dell');
        $sousCatLenovo = $this->createSousCategorie($categorieLaptops, $magasin, 'Lenovo', 'Ordinateurs Lenovo');
        $sousCatMacBook = $this->createSousCategorie($categorieLaptops, $magasin, 'MacBook', 'MacBook Apple');
        $sousCatAsus = $this->createSousCategorie($categorieLaptops, $magasin, 'ASUS', 'Ordinateurs ASUS');

        // Caractéristiques pour ordinateurs portables
        $this->createCaracteristique($sousCatHP, $magasin, 'Modèle', Caracteristique::TYPE_TEXT, null, true);
        $this->createCaracteristique($sousCatHP, $magasin, 'Processeur', Caracteristique::TYPE_SELECT, ['Intel Core i3', 'Intel Core i5', 'Intel Core i7', 'Intel Core i9', 'AMD Ryzen 3', 'AMD Ryzen 5', 'AMD Ryzen 7', 'AMD Ryzen 9'], true);
        $this->createCaracteristique($sousCatHP, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['4 GB', '8 GB', '16 GB', '32 GB', '64 GB'], true);
        $this->createCaracteristique($sousCatHP, $magasin, 'Stockage', Caracteristique::TYPE_SELECT, ['256 GB SSD', '512 GB SSD', '1 TB SSD', '256 GB HDD', '512 GB HDD', '1 TB HDD', '2 TB HDD'], true);
        $this->createCaracteristique($sousCatHP, $magasin, 'Taille écran', Caracteristique::TYPE_SELECT, ['13.3"', '14"', '15.6"', '17.3"'], true);
        $this->createCaracteristique($sousCatHP, $magasin, 'Carte graphique', Caracteristique::TYPE_SELECT, ['Intégrée', 'NVIDIA GeForce GTX', 'NVIDIA GeForce RTX', 'AMD Radeon'], false);
        $this->createCaracteristique($sousCatHP, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon', 'Occasion - Moyen'], true);

        $this->createCaracteristique($sousCatDell, $magasin, 'Modèle', Caracteristique::TYPE_TEXT, null, true);
        $this->createCaracteristique($sousCatDell, $magasin, 'Processeur', Caracteristique::TYPE_SELECT, ['Intel Core i3', 'Intel Core i5', 'Intel Core i7', 'Intel Core i9', 'AMD Ryzen 3', 'AMD Ryzen 5', 'AMD Ryzen 7', 'AMD Ryzen 9'], true);
        $this->createCaracteristique($sousCatDell, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['4 GB', '8 GB', '16 GB', '32 GB', '64 GB'], true);
        $this->createCaracteristique($sousCatDell, $magasin, 'Stockage', Caracteristique::TYPE_SELECT, ['256 GB SSD', '512 GB SSD', '1 TB SSD', '256 GB HDD', '512 GB HDD', '1 TB HDD', '2 TB HDD'], true);
        $this->createCaracteristique($sousCatDell, $magasin, 'Taille écran', Caracteristique::TYPE_SELECT, ['13.3"', '14"', '15.6"', '17.3"'], true);
        $this->createCaracteristique($sousCatDell, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon'], true);

        $this->createCaracteristique($sousCatMacBook, $magasin, 'Modèle', Caracteristique::TYPE_SELECT, ['MacBook Air M1', 'MacBook Air M2', 'MacBook Air M3', 'MacBook Pro 13" M1', 'MacBook Pro 13" M2', 'MacBook Pro 14" M1', 'MacBook Pro 14" M2', 'MacBook Pro 14" M3', 'MacBook Pro 16" M1', 'MacBook Pro 16" M2', 'MacBook Pro 16" M3'], true);
        $this->createCaracteristique($sousCatMacBook, $magasin, 'Processeur', Caracteristique::TYPE_SELECT, ['Apple M1', 'Apple M1 Pro', 'Apple M1 Max', 'Apple M2', 'Apple M2 Pro', 'Apple M2 Max', 'Apple M3', 'Apple M3 Pro', 'Apple M3 Max'], true);
        $this->createCaracteristique($sousCatMacBook, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['8 GB', '16 GB', '24 GB', '32 GB', '64 GB', '96 GB', '128 GB'], true);
        $this->createCaracteristique($sousCatMacBook, $magasin, 'Stockage', Caracteristique::TYPE_SELECT, ['256 GB', '512 GB', '1 TB', '2 TB', '4 TB', '8 TB'], true);
        $this->createCaracteristique($sousCatMacBook, $magasin, 'Couleur', Caracteristique::TYPE_SELECT, ['Gris sidéral', 'Argent', 'Or'], false);
        $this->createCaracteristique($sousCatMacBook, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon'], true);

        // 3. ORDINATEURS DE BUREAU
        $categorieDesktops = $this->createCategorie($magasin, 'Ordinateurs de Bureau', 'PC de bureau et tours');
        $sousCatPCTour = $this->createSousCategorie($categorieDesktops, $magasin, 'PC Tour', 'Ordinateurs de bureau en tour');
        $sousCatAllInOne = $this->createSousCategorie($categorieDesktops, $magasin, 'All-in-One', 'Ordinateurs tout-en-un');
        $sousCatIMac = $this->createSousCategorie($categorieDesktops, $magasin, 'iMac', 'iMac Apple');

        $this->createCaracteristique($sousCatPCTour, $magasin, 'Processeur', Caracteristique::TYPE_SELECT, ['Intel Core i3', 'Intel Core i5', 'Intel Core i7', 'Intel Core i9', 'AMD Ryzen 3', 'AMD Ryzen 5', 'AMD Ryzen 7', 'AMD Ryzen 9'], true);
        $this->createCaracteristique($sousCatPCTour, $magasin, 'RAM', Caracteristique::TYPE_SELECT, ['4 GB', '8 GB', '16 GB', '32 GB', '64 GB'], true);
        $this->createCaracteristique($sousCatPCTour, $magasin, 'Stockage', Caracteristique::TYPE_SELECT, ['256 GB SSD', '512 GB SSD', '1 TB SSD', '1 TB HDD', '2 TB HDD'], true);
        $this->createCaracteristique($sousCatPCTour, $magasin, 'Carte graphique', Caracteristique::TYPE_SELECT, ['Intégrée', 'NVIDIA GeForce GTX', 'NVIDIA GeForce RTX', 'AMD Radeon'], false);
        $this->createCaracteristique($sousCatPCTour, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon'], true);

        // 4. TABLETTES
        $categorieTablettes = $this->createCategorie($magasin, 'Tablettes', 'Tablettes tactiles');
        $sousCatiPad = $this->createSousCategorie($categorieTablettes, $magasin, 'iPad', 'Tablettes Apple');
        $sousCatSamsungTab = $this->createSousCategorie($categorieTablettes, $magasin, 'Samsung Galaxy Tab', 'Tablettes Samsung');

        $this->createCaracteristique($sousCatiPad, $magasin, 'Modèle', Caracteristique::TYPE_SELECT, ['iPad Pro 12.9"', 'iPad Pro 11"', 'iPad Air', 'iPad', 'iPad mini'], true);
        $this->createCaracteristique($sousCatiPad, $magasin, 'Capacité de stockage', Caracteristique::TYPE_SELECT, ['64 GB', '128 GB', '256 GB', '512 GB', '1 TB', '2 TB'], true);
        $this->createCaracteristique($sousCatiPad, $magasin, 'Connectivité', Caracteristique::TYPE_SELECT, ['Wi-Fi', 'Wi-Fi + Cellular'], true);
        $this->createCaracteristique($sousCatiPad, $magasin, 'Couleur', Caracteristique::TYPE_SELECT, ['Gris sidéral', 'Argent', 'Or', 'Rose'], false);
        $this->createCaracteristique($sousCatiPad, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion - Excellent', 'Occasion - Bon'], true);

        // 5. ACCESSOIRES
        $categorieAccessoires = $this->createCategorie($magasin, 'Accessoires', 'Accessoires électroniques');
        $sousCatEcouteurs = $this->createSousCategorie($categorieAccessoires, $magasin, 'Écouteurs', 'Écouteurs et casques audio');
        $sousCatChargeurs = $this->createSousCategorie($categorieAccessoires, $magasin, 'Chargeurs', 'Chargeurs et câbles');
        $sousCatCoques = $this->createSousCategorie($categorieAccessoires, $magasin, 'Coques et Housses', 'Coques de protection');

        $this->createCaracteristique($sousCatEcouteurs, $magasin, 'Type', Caracteristique::TYPE_SELECT, ['Sans fil', 'Filaire', 'True Wireless'], true);
        $this->createCaracteristique($sousCatEcouteurs, $magasin, 'Marque', Caracteristique::TYPE_SELECT, ['Apple AirPods', 'Samsung Galaxy Buds', 'Sony', 'JBL', 'Beats', 'Autre'], true);
        $this->createCaracteristique($sousCatEcouteurs, $magasin, 'État', Caracteristique::TYPE_SELECT, ['Neuf', 'Reconditionné', 'Occasion'], true);

        $io->section('Création des fournisseurs');
        $fournisseurs = [];
        $fournisseurs[] = $this->createFournisseur($magasin, 'TechDistrib', Fournisseur::TYPE_DISTRIBUTEUR, ['+33612345678', '+33612345679'], 'contact@techdistrib.fr', 'Paris', 'France');
        $fournisseurs[] = $this->createFournisseur($magasin, 'MobileWorld', Fournisseur::TYPE_WHOLESALER, ['+33698765432'], 'info@mobileworld.fr', 'Lyon', 'France');
        $fournisseurs[] = $this->createFournisseur($magasin, 'Apple Store', Fournisseur::TYPE_DISTRIBUTEUR, ['+33123456789'], 'pro@apple.com', 'Paris', 'France');
        $fournisseurs[] = $this->createFournisseur($magasin, 'Samsung Electronics', Fournisseur::TYPE_FABRICANT, ['+33611111111'], 'contact@samsung.fr', 'Paris', 'France');
        $fournisseurs[] = $this->createFournisseur($magasin, 'PC Import', Fournisseur::TYPE_IMPORTATEUR, ['+33622222222'], 'contact@pcimport.fr', 'Marseille', 'France');

        $io->section('Création des clients');
        $clients = [];
        $nomsClients = ['Ahmed Benali', 'Fatima Alami', 'Mohamed Amrani', 'Aicha Bensaid', 'Youssef El Fassi', 'Sanae Idrissi', 'Karim Tazi', 'Nadia Chraibi', 'Omar Alaoui', 'Laila Berrada'];
        foreach ($nomsClients as $nom) {
            $clients[] = $this->createClient($magasin, $nom, '0' . rand(6, 7) . rand(10000000, 99999999));
        }

        $io->section('Création des produits');

        // Produits iPhone
        $produitsIphone = [
            ['iPhone 15 Pro Max', '256 GB', 'Bleu', 'Neuf', 1299.00],
            ['iPhone 15 Pro', '256 GB', 'Noir', 'Neuf', 1149.00],
            ['iPhone 15', '128 GB', 'Rose', 'Neuf', 929.00],
            ['iPhone 14 Pro Max', '256 GB', 'Violet', 'Reconditionné', 999.00],
            ['iPhone 14', '128 GB', 'Bleu', 'Occasion - Excellent', 699.00],
            ['iPhone 13', '128 GB', 'Rouge', 'Occasion - Bon', 549.00],
        ];

        foreach ($produitsIphone as $prod) {
            $this->createProduit(
                $magasin,
                $sousCatIphone,
                $prod[0] . ' ' . $prod[1] . ' - ' . $prod[2],
                $prod[4],
                $prod[4] * 1.15,
                ['Modèle' => $prod[0], 'Capacité de stockage' => $prod[1], 'Couleur' => $prod[2], 'État' => $prod[3]],
                $fournisseurs[2] // Apple Store
            );
        }

        // Produits Samsung
        $produitsSamsung = [
            ['Galaxy S24 Ultra', '256 GB', '12 GB', 'Noir', 'Neuf', 1199.00],
            ['Galaxy S24+', '256 GB', '12 GB', 'Violet', 'Neuf', 999.00],
            ['Galaxy S23 Ultra', '256 GB', '12 GB', 'Vert', 'Reconditionné', 899.00],
            ['Galaxy S23', '128 GB', '8 GB', 'Rose', 'Occasion - Excellent', 649.00],
            ['Galaxy A54', '128 GB', '8 GB', 'Bleu', 'Neuf', 399.00],
        ];

        foreach ($produitsSamsung as $prod) {
            $this->createProduit(
                $magasin,
                $sousCatSamsung,
                $prod[0] . ' ' . $prod[1] . ' / ' . $prod[2] . ' - ' . $prod[3],
                $prod[5],
                $prod[5] * 1.15,
                ['Modèle' => $prod[0], 'Capacité de stockage' => $prod[1], 'RAM' => $prod[2], 'Couleur' => $prod[3], 'État' => $prod[4]],
                $fournisseurs[3] // Samsung
            );
        }

        // Produits MacBook
        $produitsMacBook = [
            ['MacBook Pro 16" M3', '18 GB', '512 GB', 'Gris sidéral', 'Neuf', 2499.00],
            ['MacBook Pro 14" M3', '18 GB', '512 GB', 'Argent', 'Neuf', 1999.00],
            ['MacBook Air M2', '8 GB', '256 GB', 'Or', 'Neuf', 1299.00],
            ['MacBook Pro 14" M1', '16 GB', '512 GB', 'Gris sidéral', 'Reconditionné', 1499.00],
            ['MacBook Air M1', '8 GB', '256 GB', 'Argent', 'Occasion - Excellent', 899.00],
        ];

        foreach ($produitsMacBook as $prod) {
            $this->createProduit(
                $magasin,
                $sousCatMacBook,
                $prod[0] . ' ' . $prod[1] . ' / ' . $prod[2] . ' - ' . $prod[3],
                $prod[5],
                $prod[5] * 1.12,
                ['Modèle' => $prod[0], 'RAM' => $prod[1], 'Stockage' => $prod[2], 'Couleur' => $prod[3], 'État' => $prod[4]],
                $fournisseurs[2] // Apple Store
            );
        }

        // Produits HP
        $produitsHP = [
            ['HP Pavilion 15', 'Intel Core i5', '8 GB', '512 GB SSD', '15.6"', 'Intégrée', 'Neuf', 699.00],
            ['HP EliteBook 840', 'Intel Core i7', '16 GB', '512 GB SSD', '14"', 'Intégrée', 'Reconditionné', 899.00],
            ['HP Victus 16', 'AMD Ryzen 7', '16 GB', '1 TB SSD', '16.1"', 'NVIDIA GeForce RTX', 'Neuf', 1299.00],
        ];

        foreach ($produitsHP as $prod) {
            $this->createProduit(
                $magasin,
                $sousCatHP,
                $prod[0] . ' - ' . $prod[1] . ' / ' . $prod[2] . ' / ' . $prod[3],
                $prod[7],
                $prod[7] * 1.15,
                ['Modèle' => $prod[0], 'Processeur' => $prod[1], 'RAM' => $prod[2], 'Stockage' => $prod[3], 'Taille écran' => $prod[4], 'Carte graphique' => $prod[5], 'État' => $prod[6]],
                $fournisseurs[4] // PC Import
            );
        }

        // Produits iPad
        $produitsiPad = [
            ['iPad Pro 12.9"', '256 GB', 'Wi-Fi', 'Gris sidéral', 'Neuf', 1199.00],
            ['iPad Air', '128 GB', 'Wi-Fi + Cellular', 'Rose', 'Neuf', 699.00],
            ['iPad', '64 GB', 'Wi-Fi', 'Argent', 'Reconditionné', 399.00],
        ];

        foreach ($produitsiPad as $prod) {
            $this->createProduit(
                $magasin,
                $sousCatiPad,
                $prod[0] . ' ' . $prod[1] . ' - ' . $prod[2] . ' - ' . $prod[3],
                $prod[5],
                $prod[5] * 1.15,
                ['Modèle' => $prod[0], 'Capacité de stockage' => $prod[1], 'Connectivité' => $prod[2], 'Couleur' => $prod[3], 'État' => $prod[4]],
                $fournisseurs[2] // Apple Store
            );
        }

        // Produits accessoires
        $produitsAccessoires = [
            ['AirPods Pro (2ème génération)', 'Sans fil', 'Apple AirPods', 'Neuf', 249.00, $sousCatEcouteurs],
            ['Galaxy Buds2 Pro', 'True Wireless', 'Samsung Galaxy Buds', 'Neuf', 199.00, $sousCatEcouteurs],
            ['Chargeur MagSafe', 'Chargeurs', 'Apple', 'Neuf', 39.00, $sousCatChargeurs],
        ];

        foreach ($produitsAccessoires as $prod) {
            $this->createProduit(
                $magasin,
                $prod[5],
                $prod[0],
                $prod[4],
                $prod[4] * 1.20,
                ['Type' => $prod[1], 'Marque' => $prod[2], 'État' => $prod[3]],
                $fournisseurs[0] // TechDistrib
            );
        }

        // Flush pour s'assurer que tous les produits sont persistés
        $this->em->flush();

        $io->section('Création des achats');
        // Récupérer un utilisateur pour les achats
        $utilisateur = $this->em->getRepository(User::class)->findOneBy(['magasin' => $magasin]);
        if (!$utilisateur) {
            $io->warning('Aucun utilisateur trouvé. Les achats ne seront pas créés.');
        } else {
            // Créer des achats avec différents états de paiement
            $produits = $this->em->getRepository(Produit::class)->findBy(['magasin' => $magasin], null, 15);
            $achatIndex = 0;
            foreach ($produits as $produit) {
                if ($produit->getFournisseur()) {
                    $achat = new Achat();
                    $achat->setMagasin($magasin);
                    $achat->setFournisseur($produit->getFournisseur());
                    $achat->setUtilisateur($utilisateur);
                    $achat->setNumeroFacture($this->factureService->genererNumeroFactureAchat());
                    $achat->setDateAchat($produit->getDateAchat() ?? new \DateTime('-' . rand(1, 60) . ' days'));
                    $montantTotal = (float)$produit->getPrixAchat();
                    $achat->setMontantTotal((string)$montantTotal);
                    
                    // Créer une ligne d'achat
                    $ligneAchat = new LigneAchat();
                    $ligneAchat->setAchat($achat);
                    $ligneAchat->setProduit($produit);
                    $ligneAchat->setNomProduit($produit->getNom());
                    $ligneAchat->setQuantite(1);
                    $ligneAchat->setPrixUnitaire($produit->getPrixAchat());
                    $ligneAchat->setMontantTotal($produit->getPrixAchat());
                    $achat->addLigneAchat($ligneAchat);
                    
                    // Simuler différents états de paiement : 30% impayés, 40% partiels, 30% payés
                    $etatPaiement = $achatIndex % 10;
                    $montantPaye = 0;
                    
                    if ($etatPaiement < 3) {
                        // Impayé (30%)
                        $achat->setMontantRestant((string)$montantTotal);
                    } elseif ($etatPaiement < 7) {
                        // Paiement partiel (40%)
                        $montantPaye = $montantTotal * (0.3 + (rand(0, 50) / 100)); // Entre 30% et 80%
                        $paiement = new Paiement();
                        $paiement->setFournisseur($produit->getFournisseur());
                        $paiement->setAchat($achat);
                        $paiement->setUtilisateur($utilisateur);
                        $paiement->setMontant((string)$montantPaye);
                        $paiement->setMode(Paiement::MODE_ESPECES);
                        $paiement->setDatePaiement(new \DateTime('-' . rand(1, 30) . ' days'));
                        $achat->addPaiement($paiement);
                        $this->em->persist($paiement);
                        $achat->setMontantRestant((string)($montantTotal - $montantPaye));
                    } else {
                        // Payé (30%)
                        $paiement = new Paiement();
                        $paiement->setFournisseur($produit->getFournisseur());
                        $paiement->setAchat($achat);
                        $paiement->setUtilisateur($utilisateur);
                        $paiement->setMontant((string)$montantTotal);
                        $paiement->setMode(Paiement::MODE_ESPECES);
                        $paiement->setDatePaiement(new \DateTime('-' . rand(1, 60) . ' days'));
                        $achat->addPaiement($paiement);
                        $this->em->persist($paiement);
                        $achat->setMontantRestant('0.00');
                    }
                    
                    $this->em->persist($achat);
                    $produit->setAchatFournisseur($achat);
                    $this->em->flush(); // Flush après chaque achat pour générer des numéros uniques
                    $achatIndex++;
                }
            }
        }

        $io->section('Création des ventes');
        // Créer des ventes avec différents états de paiement
        $vendeur = $this->em->getRepository(User::class)->findOneBy(['magasin' => $magasin]);
        if ($vendeur) {
            $produitsVendus = $this->em->getRepository(Produit::class)->findBy(['magasin' => $magasin, 'statut' => Produit::STATUT_EN_STOCK], null, 10);
            $venteIndex = 0;
            foreach ($produitsVendus as $produit) {
                $client = $clients[array_rand($clients)];
                $prixVente = $produit->getPrixVenteRecommande() ?? (float)$produit->getPrixAchat() * 1.15;
                
                $vente = new Vente();
                $vente->setMagasin($magasin);
                $vente->setClient($client);
                $vente->setVendeur($vendeur);
                $vente->setNumeroFacture($this->factureService->genererNumeroFacture());
                $vente->setDateVente(new \DateTime('-' . rand(1, 30) . ' days'));
                $vente->setMontantTotal((string)$prixVente);
                $vente->setReduction('0.00');
                $vente->setMontantFinal((string)$prixVente);
                
                // Créer une ligne de vente
                $ligneVente = new LigneVente();
                $ligneVente->setVente($vente);
                $ligneVente->setProduit($produit);
                $ligneVente->setQuantite(1);
                $ligneVente->setPrixUnitaire((string)$prixVente);
                $ligneVente->setPrixAchatUnitaire($produit->getPrixAchat());
                $ligneVente->setMontantTotal((string)$prixVente);
                $vente->addLigneVente($ligneVente);
                
                // Simuler différents états de paiement : 30% impayés, 40% partiels, 30% payés
                $etatPaiement = $venteIndex % 10;
                $montantPaye = 0;
                
                if ($etatPaiement < 3) {
                    // Impayé (30%)
                    $vente->setModePaiement(Vente::MODE_PAIEMENT_CREDIT);
                    $vente->setStatutPaiement(Vente::STATUT_PAIEMENT_IMPAYE);
                    $vente->setMontantPaye('0.00');
                    $vente->setMontantRestant((string)$prixVente);
                    $vente->setDateLimitePaiement(new \DateTime('+' . rand(7, 30) . ' days'));
                } elseif ($etatPaiement < 7) {
                    // Paiement partiel (40%)
                    $montantPaye = $prixVente * (0.2 + (rand(0, 60) / 100)); // Entre 20% et 80%
                    $paiementClient = new PaiementClient();
                    $paiementClient->setVente($vente);
                    $paiementClient->setClient($client);
                    $paiementClient->setUtilisateur($vendeur);
                    $paiementClient->setMontant((string)$montantPaye);
                    $paiementClient->setMode(PaiementClient::MODE_CASH);
                    $paiementClient->setDatePaiement(new \DateTime('-' . rand(1, 20) . ' days'));
                    $vente->addPaiementClient($paiementClient);
                    $this->em->persist($paiementClient);
                    
                    $vente->setModePaiement(Vente::MODE_PAIEMENT_MIXTE);
                    $vente->setMontantPaye((string)$montantPaye);
                    $vente->setMontantRestant((string)($prixVente - $montantPaye));
                    $vente->mettreAJourStatutPaiement();
                    $vente->setDateLimitePaiement(new \DateTime('+' . rand(7, 30) . ' days'));
                } else {
                    // Payé (30%)
                    $paiementClient = new PaiementClient();
                    $paiementClient->setVente($vente);
                    $paiementClient->setClient($client);
                    $paiementClient->setUtilisateur($vendeur);
                    $paiementClient->setMontant((string)$prixVente);
                    $paiementClient->setMode(PaiementClient::MODE_CASH);
                    $paiementClient->setDatePaiement(new \DateTime('-' . rand(1, 30) . ' days'));
                    $vente->addPaiementClient($paiementClient);
                    $this->em->persist($paiementClient);
                    
                    $vente->setModePaiement(Vente::MODE_PAIEMENT_CASH);
                    $vente->setMontantPaye((string)$prixVente);
                    $vente->setMontantRestant('0.00');
                    $vente->mettreAJourStatutPaiement();
                }
                
                $this->em->persist($vente);
                $this->em->persist($ligneVente);
                
                // Mettre à jour le produit
                $produit->setStatut(Produit::STATUT_VENDU);
                $produit->setPrixVente((string)$prixVente);
                $this->em->flush(); // Flush après chaque vente pour générer des numéros uniques
                $venteIndex++;
            }
        }

        $this->em->flush();

        $io->success('Données de test créées avec succès!');
        $io->table(
            ['Type', 'Nombre'],
            [
                ['Catégories', $this->em->getRepository(Categorie::class)->count(['magasin' => $magasin])],
                ['Sous-catégories', $this->em->getRepository(SousCategorie::class)->count(['magasin' => $magasin])],
                ['Caractéristiques', $this->em->getRepository(Caracteristique::class)->count(['magasin' => $magasin])],
                ['Produits', $this->em->getRepository(Produit::class)->count(['magasin' => $magasin])],
                ['Fournisseurs', $this->em->getRepository(Fournisseur::class)->count(['magasin' => $magasin])],
                ['Clients', $this->em->getRepository(Client::class)->count(['magasin' => $magasin])],
                ['Achats', $this->em->getRepository(Achat::class)->count(['magasin' => $magasin])],
                ['Ventes', $this->em->getRepository(Vente::class)->count(['magasin' => $magasin])],
            ]
        );

        return Command::SUCCESS;
    }

    private function createCategorie(Magasin $magasin, string $nom, string $description): Categorie
    {
        $categorie = new Categorie();
        $categorie->setNom($nom);
        $categorie->setDescription($description);
        $categorie->setMagasin($magasin);
        $this->em->persist($categorie);
        return $categorie;
    }

    private function createSousCategorie(Categorie $categorie, Magasin $magasin, string $nom, string $description): SousCategorie
    {
        $sousCategorie = new SousCategorie();
        $sousCategorie->setNom($nom);
        $sousCategorie->setDescription($description);
        $sousCategorie->setCategorie($categorie);
        $sousCategorie->setMagasin($magasin);
        $this->em->persist($sousCategorie);
        return $sousCategorie;
    }

    private function createCaracteristique(?SousCategorie $sousCategorie, Magasin $magasin, string $nom, string $type, ?array $valeursPossibles, bool $obligatoire): Caracteristique
    {
        $caracteristique = new Caracteristique();
        $caracteristique->setNom($nom);
        $caracteristique->setType($type);
        $caracteristique->setObligatoire($obligatoire);
        $caracteristique->setMagasin($magasin);
        if ($sousCategorie) {
            $caracteristique->setSousCategorie($sousCategorie);
        }
        if ($valeursPossibles) {
            $caracteristique->setValeursPossibles($valeursPossibles);
        }
        $this->em->persist($caracteristique);
        return $caracteristique;
    }

    private function createFournisseur(Magasin $magasin, string $nom, string $type, array $telephones, ?string $email, ?string $ville, ?string $pays): Fournisseur
    {
        $fournisseur = new Fournisseur();
        $fournisseur->setNom($nom);
        $fournisseur->setType($type);
        $fournisseur->setTelephones($telephones);
        $fournisseur->setEmail($email);
        $fournisseur->setVille($ville);
        $fournisseur->setPays($pays);
        $fournisseur->setMagasin($magasin);
        $this->em->persist($fournisseur);
        return $fournisseur;
    }

    private function createClient(Magasin $magasin, string $nom, string $telephone): Client
    {
        $client = new Client();
        $client->setNom($nom);
        $client->setTelephone($telephone);
        $client->setMagasin($magasin);
        $this->em->persist($client);
        return $client;
    }

    private function createProduit(Magasin $magasin, SousCategorie $sousCategorie, string $nom, float $prixAchat, float $prixVenteRecommande, array $caracteristiques, ?Fournisseur $fournisseur): Produit
    {
        $produit = new Produit();
        $produit->setNom($nom);
        $produit->setCategorie($sousCategorie->getCategorie());
        $produit->setSousCategorie($sousCategorie);
        $produit->setMagasin($magasin);
        $produit->setPrixAchat((string)$prixAchat);
        $produit->setPrixVenteRecommande((string)$prixVenteRecommande);
        $produit->setMargeMinimale((string)($prixVenteRecommande - $prixAchat));
        $produit->setCodeBarres('BC' . str_pad((string)rand(1000000000, 9999999999), 10, '0', STR_PAD_LEFT));
        
        // Générer un IMEI pour les téléphones
        if (str_contains($sousCategorie->getNom(), 'iPhone') || str_contains($sousCategorie->getNom(), 'Samsung') || str_contains($sousCategorie->getNom(), 'Xiaomi') || str_contains($sousCategorie->getNom(), 'Huawei')) {
            $produit->setImei(str_pad((string)rand(100000000000000, 999999999999999), 15, '0', STR_PAD_LEFT));
        }
        
        $produit->setCaracteristiquesValeurs($caracteristiques);
        $produit->setFournisseur($fournisseur);
        $produit->setOrigineAchat($fournisseur ? Produit::ORIGINE_FOURNISSEUR : Produit::ORIGINE_PARTICULIER);
        $produit->setStatut(Produit::STATUT_EN_STOCK);
        $produit->setDateAchat(new \DateTime('-' . rand(1, 60) . ' days'));
        $produit->setDateEntree($produit->getDateAchat());
        
        $this->em->persist($produit);
        return $produit;
    }

    private function deleteExistingData(Magasin $magasin): void
    {
        // Supprimer dans l'ordre pour respecter les contraintes de clés étrangères
        $this->em->getRepository(PaiementClient::class)->createQueryBuilder('p')
            ->delete()
            ->where('p.vente IN (SELECT v.id FROM App\Entity\Vente v WHERE v.magasin = :magasin)')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Paiement::class)->createQueryBuilder('p')
            ->delete()
            ->where('p.achat IN (SELECT a.id FROM App\Entity\Achat a WHERE a.magasin = :magasin)')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(LigneVente::class)->createQueryBuilder('l')
            ->delete()
            ->where('l.vente IN (SELECT v.id FROM App\Entity\Vente v WHERE v.magasin = :magasin)')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        // Supprimer les produits AVANT les achats (car produits référence achats)
        $this->em->getRepository(Produit::class)->createQueryBuilder('p')
            ->update()
            ->set('p.achatFournisseur', ':null')
            ->setParameter('null', null)
            ->where('p.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Produit::class)->createQueryBuilder('p')
            ->delete()
            ->where('p.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Vente::class)->createQueryBuilder('v')
            ->delete()
            ->where('v.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Achat::class)->createQueryBuilder('a')
            ->delete()
            ->where('a.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Caracteristique::class)->createQueryBuilder('c')
            ->delete()
            ->where('c.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(SousCategorie::class)->createQueryBuilder('s')
            ->delete()
            ->where('s.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Categorie::class)->createQueryBuilder('c')
            ->delete()
            ->where('c.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Client::class)->createQueryBuilder('c')
            ->delete()
            ->where('c.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->getRepository(Fournisseur::class)->createQueryBuilder('f')
            ->delete()
            ->where('f.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->getQuery()
            ->execute();

        $this->em->flush();
    }
}

