<?php

namespace App\Entity;

use App\Repository\VenteRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity(repositoryClass: VenteRepository::class)]
#[ORM\Table(name: 'ventes')]
class Vente
{
    const MODE_PAIEMENT_CASH = 'cash';
    const MODE_PAIEMENT_CARTE = 'carte';
    const MODE_PAIEMENT_VIREMENT = 'virement';
    const MODE_PAIEMENT_CHEQUE = 'cheque';
    const MODE_PAIEMENT_CREDIT = 'credit';
    const MODE_PAIEMENT_MIXTE = 'mixte';

    const STATUT_PAIEMENT_PAYE = 'paye';
    const STATUT_PAIEMENT_PARTIEL = 'partiel';
    const STATUT_PAIEMENT_IMPAYE = 'impaye';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 50, unique: true)]
    private ?string $numeroFacture = null;

    #[ORM\ManyToOne]
    #[ORM\JoinColumn(nullable: false)]
    private ?User $vendeur = null;

    #[ORM\ManyToOne(inversedBy: 'ventes')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Magasin $magasin = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantTotal = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
    private ?string $reduction = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantFinal = null;

    #[ORM\Column(length: 50)]
    private ?string $modePaiement = self::MODE_PAIEMENT_CASH;

    #[ORM\ManyToOne(inversedBy: 'ventes')]
    #[ORM\JoinColumn(nullable: true)]
    private ?Client $client = null; // Client peut être null pour "client rapide"

    #[ORM\Column(length: 50)]
    private ?string $statutPaiement = self::STATUT_PAIEMENT_PAYE;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantPaye = null; // Montant total payé

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $montantRestant = null; // Montant restant à payer

    #[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
    private ?\DateTimeInterface $dateLimitePaiement = null; // Date limite si crédit

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateVente = null;

    #[ORM\OneToMany(mappedBy: 'vente', targetEntity: LigneVente::class, cascade: ['persist', 'remove'])]
    private Collection $lignesVente;

    #[ORM\OneToMany(mappedBy: 'vente', targetEntity: PaiementClient::class, cascade: ['persist', 'remove'])]
    private Collection $paiementsClient;

    #[ORM\Column(type: Types::BOOLEAN)]
    private bool $retour = false;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $notes = null;

    public function __construct()
    {
        $this->lignesVente = new ArrayCollection();
        $this->paiementsClient = new ArrayCollection();
        $this->dateVente = new \DateTime();
        $this->reduction = '0.00';
        $this->montantPaye = '0.00';
        $this->montantRestant = '0.00';
        $this->statutPaiement = self::STATUT_PAIEMENT_PAYE;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getNumeroFacture(): ?string
    {
        return $this->numeroFacture;
    }

    public function setNumeroFacture(string $numeroFacture): self
    {
        $this->numeroFacture = $numeroFacture;
        return $this;
    }

    public function getVendeur(): ?User
    {
        return $this->vendeur;
    }

    public function setVendeur(?User $vendeur): self
    {
        $this->vendeur = $vendeur;
        return $this;
    }

    public function getMagasin(): ?Magasin
    {
        return $this->magasin;
    }

    public function setMagasin(?Magasin $magasin): self
    {
        $this->magasin = $magasin;
        return $this;
    }

    public function getMontantTotal(): ?string
    {
        return $this->montantTotal;
    }

    public function setMontantTotal(string $montantTotal): self
    {
        $this->montantTotal = $montantTotal;
        return $this;
    }

    public function getReduction(): ?string
    {
        return $this->reduction;
    }

    public function setReduction(?string $reduction): self
    {
        $this->reduction = $reduction;
        return $this;
    }

    public function getMontantFinal(): ?string
    {
        return $this->montantFinal;
    }

    public function setMontantFinal(string $montantFinal): self
    {
        $this->montantFinal = $montantFinal;
        return $this;
    }

    public function getModePaiement(): ?string
    {
        return $this->modePaiement;
    }

    public function setModePaiement(string $modePaiement): self
    {
        $this->modePaiement = $modePaiement;
        return $this;
    }

    public function getDateVente(): ?\DateTimeInterface
    {
        return $this->dateVente;
    }

    public function setDateVente(\DateTimeInterface $dateVente): self
    {
        $this->dateVente = $dateVente;
        return $this;
    }

    /**
     * @return Collection<int, LigneVente>
     */
    public function getLignesVente(): Collection
    {
        return $this->lignesVente;
    }

    public function addLigneVente(LigneVente $ligneVente): self
    {
        if (!$this->lignesVente->contains($ligneVente)) {
            $this->lignesVente->add($ligneVente);
            $ligneVente->setVente($this);
        }
        return $this;
    }

    public function isRetour(): bool
    {
        return $this->retour;
    }

    public function setRetour(bool $retour): self
    {
        $this->retour = $retour;
        return $this;
    }

    public function getNotes(): ?string
    {
        return $this->notes;
    }

    public function setNotes(?string $notes): self
    {
        $this->notes = $notes;
        return $this;
    }

    public function getMargeTotale(): float
    {
        $marge = 0;
        foreach ($this->lignesVente as $ligne) {
            $marge += $ligne->getMarge();
        }
        return $marge;
    }

    public function getClient(): ?Client
    {
        return $this->client;
    }

    public function setClient(?Client $client): self
    {
        $this->client = $client;
        return $this;
    }

    public function getStatutPaiement(): ?string
    {
        return $this->statutPaiement;
    }

    public function setStatutPaiement(string $statutPaiement): self
    {
        $this->statutPaiement = $statutPaiement;
        return $this;
    }

    public function getMontantPaye(): ?string
    {
        return $this->montantPaye;
    }

    public function setMontantPaye(string $montantPaye): self
    {
        $this->montantPaye = $montantPaye;
        return $this;
    }

    public function getMontantRestant(): ?string
    {
        return $this->montantRestant;
    }

    public function setMontantRestant(string $montantRestant): self
    {
        $this->montantRestant = $montantRestant;
        return $this;
    }

    public function getDateLimitePaiement(): ?\DateTimeInterface
    {
        return $this->dateLimitePaiement;
    }

    public function setDateLimitePaiement(?\DateTimeInterface $dateLimitePaiement): self
    {
        $this->dateLimitePaiement = $dateLimitePaiement;
        return $this;
    }

    /**
     * @return Collection<int, PaiementClient>
     */
    public function getPaiementsClient(): Collection
    {
        return $this->paiementsClient;
    }

    public function addPaiementClient(PaiementClient $paiementClient): self
    {
        if (!$this->paiementsClient->contains($paiementClient)) {
            $this->paiementsClient->add($paiementClient);
            $paiementClient->setVente($this);
        }
        return $this;
    }

    public function removePaiementClient(PaiementClient $paiementClient): self
    {
        if ($this->paiementsClient->removeElement($paiementClient)) {
            if ($paiementClient->getVente() === $this) {
                $paiementClient->setVente(null);
            }
        }
        return $this;
    }

    /**
     * Calcule le montant total payé à partir des paiements
     */
    public function calculerMontantPaye(): float
    {
        $total = 0.0;
        foreach ($this->paiementsClient as $paiement) {
            $total += (float)$paiement->getMontant();
        }
        return $total;
    }

    /**
     * Met à jour le statut de paiement en fonction des montants
     */
    public function mettreAJourStatutPaiement(): void
    {
        $montantPaye = $this->calculerMontantPaye();
        $this->montantPaye = (string)$montantPaye;
        $this->montantRestant = (string)((float)$this->montantFinal - $montantPaye);

        if ($montantPaye >= (float)$this->montantFinal) {
            $this->statutPaiement = self::STATUT_PAIEMENT_PAYE;
        } elseif ($montantPaye > 0) {
            $this->statutPaiement = self::STATUT_PAIEMENT_PARTIEL;
        } else {
            $this->statutPaiement = self::STATUT_PAIEMENT_IMPAYE;
        }
    }
}

