<?php

namespace App\Form;

use App\Entity\Produit;
use App\Entity\Categorie;
use App\Entity\SousCategorie;
use App\Entity\Fournisseur;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\MoneyType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

class ProduitType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $magasin = $options['magasin'];
        $produit = $options['data'] ?? null;

        $builder
            ->add('categorie', EntityType::class, [
                'class' => Categorie::class,
                'choices' => $magasin->getCategories(),
                'choice_label' => 'nom',
                'label' => 'Catégorie',
                'attr' => [
                    'class' => 'form-select',
                    'data-produit-categorie' => true
                ],
                'placeholder' => 'Sélectionnez une catégorie'
            ])
            ->add('sousCategorie', EntityType::class, [
                'class' => SousCategorie::class,
                'choices' => $produit && $produit->getCategorie() 
                    ? $produit->getCategorie()->getSousCategories() 
                    : [],
                'choice_label' => 'nom',
                'label' => 'Sous-catégorie',
                'required' => true,
                'attr' => [
                    'class' => 'form-select',
                    'data-produit-sous-categorie' => true
                ],
                'placeholder' => 'Sélectionnez d\'abord une catégorie'
            ])
            ->add('nom', TextType::class, [
                'label' => 'Nom du produit',
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'Le nom sera auto-généré si laissé vide'
                ],
                'required' => false
            ])
            ->add('referenceInterne', TextType::class, [
                'label' => 'Référence interne (générée automatiquement)',
                'required' => false,
                'disabled' => true,
                'attr' => [
                    'class' => 'form-control',
                    'readonly' => true
                ]
            ])
            ->add('imei', TextType::class, [
                'label' => 'IMEI (obligatoire pour téléphones)',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'Ex: 123456789012345'
                ]
            ])
            ->add('numeroSerie', TextType::class, [
                'label' => 'Numéro de série',
                'required' => false,
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('description', TextareaType::class, [
                'label' => 'Description interne',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'rows' => 4
                ]
            ])
            ->add('prixAchat', MoneyType::class, [
                'label' => 'Prix d\'achat',
                'currency' => 'MAD',
                'scale' => 2,
                'attr' => [
                    'class' => 'form-control',
                    'data-prix-achat' => true
                ]
            ])
            ->add('margeMinimale', MoneyType::class, [
                'label' => 'Marge minimale',
                'currency' => 'MAD',
                'scale' => 2,
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'data-marge-minimale' => true
                ]
            ])
            ->add('prixVenteRecommande', MoneyType::class, [
                'label' => 'Prix de vente recommandé (calculé automatiquement)',
                'currency' => 'MAD',
                'scale' => 2,
                'required' => false,
                'disabled' => true,
                'attr' => [
                    'class' => 'form-control',
                    'data-prix-vente-recommande' => true
                ]
            ])
            ->add('prixVente', MoneyType::class, [
                'label' => 'Prix de vente final',
                'currency' => 'MAD',
                'scale' => 2,
                'required' => false,
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('dateAchat', DateType::class, [
                'label' => 'Date d\'achat',
                'widget' => 'single_text',
                'required' => false,
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('statut', ChoiceType::class, [
                'label' => 'Statut',
                'choices' => [
                    'En stock' => Produit::STATUT_EN_STOCK,
                    'Réservé' => Produit::STATUT_RESERVE,
                    'Vendu' => Produit::STATUT_VENDU,
                    'En réparation' => Produit::STATUT_EN_REPARATION,
                    'Retourné' => Produit::STATUT_RETOURNE,
                    'Perdu' => Produit::STATUT_PERDU,
                    'Transféré' => Produit::STATUT_TRANSFERE,
                ],
                'attr' => [
                    'class' => 'form-select'
                ]
            ])
            ->add('origineAchat', ChoiceType::class, [
                'label' => 'Origine d\'achat',
                'choices' => [
                    'Fournisseur' => Produit::ORIGINE_FOURNISSEUR,
                    'Particulier' => Produit::ORIGINE_PARTICULIER,
                ],
                'attr' => [
                    'class' => 'form-select',
                    'data-origine-achat' => true
                ]
            ])
            ->add('fournisseur', EntityType::class, [
                'class' => Fournisseur::class,
                'choices' => $magasin->getFournisseurs()->filter(fn($f) => $f->isActif()),
                'choice_label' => 'nom',
                'required' => false,
                'attr' => [
                    'class' => 'form-select',
                    'data-fournisseur-field' => true
                ],
                'placeholder' => 'Sélectionnez un fournisseur'
            ]);

        // Mettre à jour les sous-catégories quand la catégorie change
        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
            $produit = $event->getData();
            $form = $event->getForm();
            
            if ($produit && $produit->getCategorie()) {
                $form->get('sousCategorie')->setData(null);
            }
        });

        $builder->get('categorie')->addEventListener(FormEvents::POST_SUBMIT, function (FormEvent $event) {
            $form = $event->getForm();
            $categorie = $form->getData();
            
            if ($categorie) {
                $form->getParent()->add('sousCategorie', EntityType::class, [
                    'class' => SousCategorie::class,
                    'choices' => $categorie->getSousCategories(),
                    'choice_label' => 'nom',
                    'label' => 'Sous-catégorie',
                    'required' => true,
                    'attr' => [
                        'class' => 'form-select',
                        'data-produit-sous-categorie' => true
                    ],
                    'placeholder' => 'Sélectionnez une sous-catégorie'
                ]);
            }
        });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Produit::class,
            'magasin' => null,
        ]);
    }
}

