# Checklist de Préparation du Projet pour le Serveur

Ce document résume toutes les étapes nécessaires pour préparer le projet avant le déploiement sur le serveur.

## ✅ Fichiers créés pour le déploiement

### 1. Fichiers de configuration
- [x] `docker-compose.prod.yml` - Configuration Docker pour la production
- [x] `Dockerfile.prod` - Dockerfile optimisé pour la production
- [x] `ENV_EXAMPLE.txt` - Exemple de variables d'environnement
- [x] `docker/nginx/nginx.prod.conf` - Configuration Nginx pour la production

### 2. Scripts
- [x] `deploy.sh` - Script de déploiement automatique (avec Docker)
- [x] `deploy-prod.sh` - Script de déploiement automatique (SANS Docker)
- [x] `backup.sh` - Script de sauvegarde de la base de données

### 3. Documentation
- [x] `DEPLOY.md` - Guide complet de déploiement avec Docker
- [x] `DEPLOY_SANS_DOCKER.md` - Guide complet de déploiement SANS Docker
- [x] `PREPARATION_SERVEUR.md` - Ce fichier (checklist)

## 📋 Avant de déployer sur le serveur

### 1. Variables d'environnement à configurer

Créez un fichier `.env` à partir de `ENV_EXAMPLE.txt` et configurez :

- [ ] `APP_SECRET` - Générer une valeur aléatoire sécurisée
- [ ] `DATABASE_URL` - URL de connexion à la base de données
- [ ] `MYSQL_ROOT_PASSWORD` - Mot de passe root MySQL (fort et sécurisé)
- [ ] `MYSQL_PASSWORD` - Mot de passe utilisateur MySQL (fort et sécurisé)
- [ ] `JWT_PASSPHRASE` - Passphrase pour JWT (fort et sécurisé)

### 2. Génération des secrets

```bash
# Générer APP_SECRET
php -r "echo bin2hex(random_bytes(32)) . PHP_EOL;"

# Générer JWT keys (si nécessaire)
mkdir -p config/jwt
php bin/console lexik:jwt:generate-keypair --skip-if-exists
```

### 3. Fichiers à vérifier

- [ ] `.env` - Configuré avec les bonnes valeurs
- [ ] `docker-compose.prod.yml` - Ports et configurations adaptés
- [ ] `docker/nginx/nginx.prod.conf` - Nom de domaine configuré si nécessaire
- [ ] Permissions des fichiers (`chmod +x deploy.sh backup.sh`)

### 4. Base de données

- [ ] Vérifier que le serveur MySQL/MariaDB est accessible
- [ ] Vérifier les identifiants de connexion
- [ ] Tester la connexion si possible

### 5. Serveur

#### Prérequis système
- [ ] Docker installé (version 20.10+)
- [ ] Docker Compose installé (version 1.29+)
- [ ] Au moins 2GB de RAM disponible
- [ ] Au moins 10GB d'espace disque libre

#### Configuration réseau
- [ ] Port 80 ouvert (HTTP)
- [ ] Port 443 ouvert (HTTPS, si SSL configuré)
- [ ] Firewall configuré correctement

### 6. Sécurité

- [ ] Tous les mots de passe par défaut ont été changés
- [ ] Mots de passe forts utilisés
- [ ] Firewall activé et configuré
- [ ] SSL/HTTPS configuré (Let's Encrypt recommandé)
- [ ] Accès SSH sécurisé (clés SSH recommandées)

### 7. Sauvegardes

- [ ] Script de sauvegarde configuré (`backup.sh`)
- [ ] Cron job configuré pour sauvegardes automatiques
- [ ] Répertoire de sauvegarde créé (`/var/backups/facture`)

## 🚀 Étapes de déploiement

### Option 1 : Avec Docker

#### 1. Transférer les fichiers sur le serveur

```bash
# Depuis votre machine locale
scp -r /path/to/facture user@server:/var/www/
```

Ou utiliser Git :

```bash
# Sur le serveur
cd /var/www
git clone <votre-repository> facture
cd facture
```

#### 2. Configurer l'environnement

```bash
cd /var/www/facture
cp ENV_EXAMPLE.txt .env
nano .env  # Éditer avec vos valeurs
```

#### 3. Lancer le déploiement

```bash
chmod +x deploy.sh backup.sh
./deploy.sh
```

#### 4. Créer l'utilisateur administrateur

```bash
docker-compose -f docker-compose.prod.yml exec php php bin/console app:create-admin
```

### Option 2 : SANS Docker (Installation directe)

#### 1. Installer les dépendances système

```bash
# Voir DEPLOY_SANS_DOCKER.md pour les instructions complètes
sudo apt install php8.2 php8.2-fpm mysql-server nginx composer
```

#### 2. Transférer les fichiers sur le serveur

```bash
# Sur le serveur
cd /var/www
git clone <votre-repository> facture
cd facture
```

#### 3. Configurer l'environnement

```bash
cd /var/www/facture
cp ENV_EXAMPLE.txt .env
nano .env  # Éditer avec vos valeurs (DATABASE_URL doit pointer vers localhost)
```

#### 4. Installer les dépendances et configurer

```bash
composer install --no-dev --optimize-autoloader
php bin/console lexik:jwt:generate-keypair --skip-if-exists
```

#### 5. Configurer Nginx

```bash
# Voir DEPLOY_SANS_DOCKER.md pour la configuration complète
sudo nano /etc/nginx/sites-available/facture
sudo ln -s /etc/nginx/sites-available/facture /etc/nginx/sites-enabled/
sudo systemctl restart nginx
```

#### 6. Lancer le déploiement

```bash
chmod +x deploy-prod.sh backup.sh
./deploy-prod.sh
```

#### 7. Créer l'utilisateur administrateur

```bash
php bin/console app:create-admin
```

### 5. Configurer les sauvegardes automatiques

```bash
# Éditer le crontab
crontab -e

# Ajouter cette ligne pour une sauvegarde quotidienne à 2h du matin
0 2 * * * /var/www/facture/backup.sh >> /var/log/facture-backup.log 2>&1
```

## 📝 Après le déploiement

### Vérifications

- [ ] L'application est accessible via le navigateur
- [ ] Le login fonctionne
- [ ] Les migrations de base de données sont appliquées
- [ ] Les logs ne contiennent pas d'erreurs critiques
- [ ] Les uploads de fichiers fonctionnent

### Tests

- [ ] Créer un produit
- [ ] Créer une vente
- [ ] Télécharger une facture
- [ ] Vérifier le tableau de bord

### Monitoring

- [ ] Configurer la surveillance des logs
- [ ] Vérifier l'espace disque régulièrement
- [ ] Vérifier que les sauvegardes se créent correctement

## 🔧 Maintenance

### Mise à jour de l'application

```bash
cd /var/www/facture
git pull
./deploy.sh
```

### Voir les logs

```bash
docker-compose -f docker-compose.prod.yml logs -f
```

### Redémarrer les services

```bash
docker-compose -f docker-compose.prod.yml restart
```

## 📞 Support

En cas de problème :
1. Consulter les logs : `docker-compose -f docker-compose.prod.yml logs`
2. Vérifier le guide de déploiement : `DEPLOY.md`
3. Vérifier les permissions des fichiers
4. Vérifier la configuration de la base de données

## 📦 Fichiers à ne JAMAIS commiter

Assurez-vous que ces fichiers sont dans `.gitignore` :
- `.env`
- `.env.local`
- `config/jwt/*.pem`
- `/var/`
- `/vendor/`

## ✅ Résumé des commandes importantes

### Avec Docker

```bash
# Déploiement
./deploy.sh

# Sauvegarde
./backup.sh

# Logs
docker-compose -f docker-compose.prod.yml logs -f

# Redémarrer
docker-compose -f docker-compose.prod.yml restart

# Arrêter
docker-compose -f docker-compose.prod.yml down

# Mettre à jour
git pull && ./deploy.sh
```

### SANS Docker

```bash
# Déploiement
./deploy-prod.sh

# Sauvegarde
./backup.sh

# Logs
sudo tail -f /var/log/nginx/facture_error.log
tail -f var/log/prod.log

# Redémarrer les services
sudo systemctl restart nginx
sudo systemctl restart php8.2-fpm

# Mettre à jour
git pull && ./deploy-prod.sh
```

Bon déploiement ! 🚀

