<?php

declare(strict_types=1);

namespace DoctrineMigrations;

use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

/**
 * Migration pour ajouter les colonnes manquantes à la table produits
 */
final class Version20251203024005 extends AbstractMigration
{
    public function getDescription(): string
    {
        return 'Ajouter les colonnes manquantes à la table produits';
    }

    public function up(Schema $schema): void
    {
        // Vérifier et ajouter chaque colonne individuellement
        $columns = [
            'reference_interne' => 'VARCHAR(255) DEFAULT NULL',
            'imei' => 'VARCHAR(50) DEFAULT NULL',
            'numero_serie' => 'VARCHAR(100) DEFAULT NULL',
            'marge_minimale' => 'NUMERIC(10, 2) DEFAULT NULL',
            'prix_vente_recommandé' => 'NUMERIC(10, 2) DEFAULT NULL',
            'origine_achat' => 'VARCHAR(50) DEFAULT NULL',
            'date_achat' => 'DATETIME DEFAULT NULL',
            'achat_fournisseur_id' => 'INT DEFAULT NULL',
        ];

        foreach ($columns as $columnName => $columnDefinition) {
            $exists = $this->connection->executeQuery(
                "SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'produits' AND COLUMN_NAME = ?",
                [$columnName]
            )->fetchOne();

            if ($exists == 0) {
                $this->addSql("ALTER TABLE produits ADD COLUMN {$columnName} {$columnDefinition}");
            }
        }

        // Initialiser les valeurs pour les produits existants AVANT de rendre NOT NULL
        $this->addSql("UPDATE produits SET date_achat = COALESCE(date_achat, date_entree, NOW()) WHERE date_achat IS NULL");
        $this->addSql("UPDATE produits SET origine_achat = COALESCE(origine_achat, 'fournisseur') WHERE origine_achat IS NULL");
        
        // Rendre les colonnes NOT NULL seulement après initialisation
        $this->addSql("ALTER TABLE produits MODIFY origine_achat VARCHAR(50) NOT NULL");
        $this->addSql("ALTER TABLE produits MODIFY date_achat DATETIME NOT NULL");
        
        // Ajouter la contrainte pour achat_fournisseur_id
        $constraintExists = $this->connection->executeQuery(
            "SELECT COUNT(*) FROM information_schema.TABLE_CONSTRAINTS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'produits' AND CONSTRAINT_NAME = 'FK_BE2DDF8CEE4462DB'"
        )->fetchOne();

        if ($constraintExists == 0) {
            $this->addSql('ALTER TABLE produits ADD CONSTRAINT FK_BE2DDF8CEE4462DB FOREIGN KEY (achat_fournisseur_id) REFERENCES achats (id)');
            $this->addSql('CREATE INDEX IDX_BE2DDF8CEE4462DB ON produits (achat_fournisseur_id)');
        }

        // Ajouter l'index unique pour imei
        $indexExists = $this->connection->executeQuery(
            "SELECT COUNT(*) FROM information_schema.STATISTICS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'produits' AND INDEX_NAME = 'UNIQ_BE2DDF8CB8179F8'"
        )->fetchOne();

        if ($indexExists == 0) {
            $this->addSql('CREATE UNIQUE INDEX UNIQ_BE2DDF8CB8179F8 ON produits (imei)');
        }

        // Vérifier si la colonne statut existe, sinon renommer etat en statut
        $statutExists = $this->connection->executeQuery(
            "SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'produits' AND COLUMN_NAME = 'statut'"
        )->fetchOne();

        $etatExists = $this->connection->executeQuery(
            "SELECT COUNT(*) FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'produits' AND COLUMN_NAME = 'etat'"
        )->fetchOne();

        if ($statutExists == 0 && $etatExists > 0) {
            $this->addSql('ALTER TABLE produits CHANGE etat statut VARCHAR(50) DEFAULT NULL');
            $this->addSql("UPDATE produits SET statut = 'en_stock' WHERE statut IS NULL");
            $this->addSql('ALTER TABLE produits MODIFY statut VARCHAR(50) NOT NULL');
        } elseif ($statutExists == 0) {
            $this->addSql('ALTER TABLE produits ADD statut VARCHAR(50) DEFAULT NULL');
            $this->addSql("UPDATE produits SET statut = 'en_stock' WHERE statut IS NULL");
            $this->addSql('ALTER TABLE produits MODIFY statut VARCHAR(50) NOT NULL');
        }

        // Modifier sous_categorie_id pour être NOT NULL (seulement si tous les produits ont une sous-catégorie)
        $produitsSansSousCategorie = $this->connection->executeQuery(
            "SELECT COUNT(*) FROM produits WHERE sous_categorie_id IS NULL"
        )->fetchOne();
        
        if ($produitsSansSousCategorie > 0) {
            // Créer une sous-catégorie par défaut pour chaque catégorie qui n'en a pas
            $this->addSql("
                INSERT INTO sous_categories (nom, categorie_id, magasin_id, date_creation)
                SELECT CONCAT('Défaut - ', c.nom), c.id, c.magasin_id, NOW()
                FROM categories c
                WHERE NOT EXISTS (
                    SELECT 1 FROM sous_categories sc WHERE sc.categorie_id = c.id
                )
            ");
            
            // Assigner une sous-catégorie à chaque produit qui n'en a pas
            $this->addSql("
                UPDATE produits p
                SET sous_categorie_id = (
                    SELECT id FROM sous_categories sc 
                    WHERE sc.categorie_id = p.categorie_id 
                    LIMIT 1
                )
                WHERE p.sous_categorie_id IS NULL
            ");
        }
        
        // Maintenant on peut rendre NOT NULL
        $this->addSql('ALTER TABLE produits MODIFY sous_categorie_id INT NOT NULL');

        // Modifier prix_vente pour être nullable
        $this->addSql('ALTER TABLE produits MODIFY prix_vente NUMERIC(10, 2) DEFAULT NULL');
    }

    public function down(Schema $schema): void
    {
        // Ne pas supprimer les colonnes pour éviter de perdre des données
    }
}

