<?php

namespace App\Command;

use App\Entity\Magasin;
use App\Entity\User;
use App\Entity\Categorie;
use App\Entity\Produit;
use App\Entity\Fournisseur;
use App\Entity\Vente;
use App\Entity\LigneVente;
use App\Entity\Achat;
use App\Entity\LigneAchat;
use App\Entity\Depense;
use App\Service\CodeBarresService;
use App\Service\StockService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

#[AsCommand(
    name: 'app:generate-fake-data',
    description: 'Génère des données fictives pour tester l\'application',
)]
class GenerateFakeDataCommand extends Command
{
    private array $nomsProduits = [
        'iPhone 15 Pro Max', 'Samsung Galaxy S24 Ultra', 'MacBook Pro M3', 'iPad Air', 
        'Sony WH-1000XM5', 'AirPods Pro', 'Samsung 4K TV 55"', 'LG OLED 65"',
        'PlayStation 5', 'Xbox Series X', 'Nintendo Switch', 'Steam Deck',
        'Logitech MX Master 3', 'Razer DeathAdder', 'Corsair K70', 'HyperX Cloud II',
        'Canon EOS R6', 'Nikon D850', 'GoPro Hero 12', 'DJI Mini 4 Pro'
    ];

    private array $nomsFournisseurs = [
        'TechDistrib', 'ElectroSupply', 'GadgetPro', 'DigitalWorld', 'TechHub',
        'ElectroMax', 'SmartDevices', 'TechZone', 'DigitalStore', 'ElectroPlus'
    ];

    private array $categories = [
        'Smartphones', 'Ordinateurs', 'Tablettes', 'TV & Écrans', 'Audio',
        'Gaming', 'Accessoires', 'Photo & Vidéo', 'Drones', 'Périphériques'
    ];

    public function __construct(
        private EntityManagerInterface $em,
        private UserPasswordHasherInterface $passwordHasher,
        private CodeBarresService $codeBarresService,
        private StockService $stockService
    ) {
        parent::__construct();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $io->title('Génération de données fictives');

        // Récupérer ou créer le magasin
        $magasin = $this->em->getRepository(Magasin::class)->findOneBy([]);
        if (!$magasin) {
            $io->error('Aucun magasin trouvé. Créez d\'abord un magasin avec app:create-admin');
            return Command::FAILURE;
        }

        $io->section('Création des catégories...');
        $categories = $this->createCategories($magasin, $io);

        $io->section('Création des fournisseurs...');
        $fournisseurs = $this->createFournisseurs($magasin, $io);

        $io->section('Création des produits...');
        $produits = $this->createProduits($magasin, $categories, $fournisseurs, $io);

        $io->section('Création des achats...');
        $this->createAchats($magasin, $produits, $fournisseurs, $io);

        $io->section('Création des ventes...');
        $this->createVentes($magasin, $produits, $io);

        $io->section('Création des dépenses...');
        $this->createDepenses($magasin, $io);

        $io->success('Données fictives générées avec succès!');
        return Command::SUCCESS;
    }

    private function createCategories(Magasin $magasin, SymfonyStyle $io): array
    {
        $categories = [];
        foreach ($this->categories as $nomCategorie) {
            $categorie = new Categorie();
            $categorie->setNom($nomCategorie);
            $categorie->setDescription("Catégorie pour les produits {$nomCategorie}");
            $categorie->setMagasin($magasin);
            $this->em->persist($categorie);
            $categories[] = $categorie;
        }
        $this->em->flush();
        $io->info(sprintf('✓ %d catégories créées', count($categories)));
        return $categories;
    }

    private function createFournisseurs(Magasin $magasin, SymfonyStyle $io): array
    {
        $fournisseurs = [];
        foreach ($this->nomsFournisseurs as $nom) {
            $fournisseur = new Fournisseur();
            $fournisseur->setNom($nom);
            $fournisseur->setContact("Contact {$nom}");
            $fournisseur->setEmail(strtolower(str_replace(' ', '', $nom)) . '@example.com');
            $fournisseur->setTelephone('0' . rand(100000000, 999999999));
            $fournisseur->setAdresse(rand(1, 100) . " Rue de la Tech, Paris");
            $fournisseur->setMagasin($magasin);
            $this->em->persist($fournisseur);
            $fournisseurs[] = $fournisseur;
        }
        $this->em->flush();
        $io->info(sprintf('✓ %d fournisseurs créés', count($fournisseurs)));
        return $fournisseurs;
    }

    private function createProduits(Magasin $magasin, array $categories, array $fournisseurs, SymfonyStyle $io): array
    {
        $produits = [];
        $etats = [Produit::ETAT_NEUF, Produit::ETAT_OCCASION];
        
        foreach ($this->nomsProduits as $nomProduit) {
            $produit = new Produit();
            $produit->setNom($nomProduit);
            $produit->setDescription("Description détaillée pour {$nomProduit}");
            $produit->setCategorie($categories[array_rand($categories)]);
            $produit->setMagasin($magasin);
            $produit->setPrixAchat((string)rand(100, 2000));
            $produit->setPrixVente((string)((float)$produit->getPrixAchat() * (1 + rand(15, 40) / 100)));
            $produit->setQuantite(rand(0, 50));
            $produit->setCodeBarres($this->codeBarresService->genererCodeBarres());
            $produit->setEtat($etats[array_rand($etats)]);
            $produit->setFournisseur($fournisseurs[array_rand($fournisseurs)]);
            $produit->setDateEntree(new \DateTime('-' . rand(1, 180) . ' days'));
            
            $this->em->persist($produit);
            $produits[] = $produit;
        }
        $this->em->flush();
        $io->info(sprintf('✓ %d produits créés', count($produits)));
        return $produits;
    }

    private function createAchats(Magasin $magasin, array $produits, array $fournisseurs, SymfonyStyle $io): void
    {
        $users = $this->em->getRepository(User::class)->findBy(['magasin' => $magasin]);
        if (empty($users)) {
            $io->warning('Aucun utilisateur trouvé pour créer les achats');
            return;
        }

        for ($i = 0; $i < 20; $i++) {
            $achat = new Achat();
            $achat->setNumeroFacture('ACH-' . time() . '-' . uniqid() . '-' . $i);
            $achat->setFournisseur($fournisseurs[array_rand($fournisseurs)]);
            $achat->setMagasin($magasin);
            $achat->setUtilisateur($users[array_rand($users)]);
            $achat->setDateAchat(new \DateTime('-' . rand(1, 90) . ' days'));

            $montantTotal = 0;
            $nbLignes = rand(2, 5);
            for ($j = 0; $j < $nbLignes; $j++) {
                $produit = $produits[array_rand($produits)];
                $ligne = new LigneAchat();
                $ligne->setAchat($achat);
                $ligne->setProduit($produit);
                $ligne->setQuantite(rand(5, 20));
                $ligne->setPrixUnitaire($produit->getPrixAchat());
                $ligne->setMontantTotal((string)((float)$ligne->getPrixUnitaire() * $ligne->getQuantite()));
                $montantTotal += (float)$ligne->getMontantTotal();
                $this->em->persist($ligne);
            }

            $achat->setMontantTotal((string)$montantTotal);
            $this->em->persist($achat);
        }
        $this->em->flush();
        $io->info('✓ 20 achats créés');
    }

    private function createVentes(Magasin $magasin, array $produits, SymfonyStyle $io): void
    {
        $users = $this->em->getRepository(User::class)->findBy(['magasin' => $magasin]);
        if (empty($users)) {
            $io->warning('Aucun utilisateur trouvé pour créer les ventes');
            return;
        }

        $modesPaiement = [Vente::MODE_PAIEMENT_CASH, Vente::MODE_PAIEMENT_CARTE, Vente::MODE_PAIEMENT_TRANSFERT];

        for ($i = 0; $i < 50; $i++) {
            $vente = new Vente();
            $vente->setNumeroFacture('FAC-' . time() . '-' . uniqid() . '-' . $i);
            $vente->setVendeur($users[array_rand($users)]);
            $vente->setMagasin($magasin);
            $vente->setDateVente(new \DateTime('-' . rand(0, 30) . ' days'));
            $vente->setModePaiement($modesPaiement[array_rand($modesPaiement)]);
            $vente->setReduction((string)rand(0, 50));

            $montantTotal = 0;
            $nbLignes = rand(1, 4);
            $lignes = [];
            
            for ($j = 0; $j < $nbLignes; $j++) {
                $produit = $produits[array_rand($produits)];
                if ($produit->getQuantite() <= 0) continue;

                $ligne = new LigneVente();
                $ligne->setVente($vente);
                $ligne->setProduit($produit);
                $ligne->setQuantite(rand(1, min(3, $produit->getQuantite())));
                $ligne->setPrixUnitaire($produit->getPrixVente());
                $ligne->setPrixAchatUnitaire($produit->getPrixAchat());
                $ligne->setMontantTotal((string)((float)$ligne->getPrixUnitaire() * $ligne->getQuantite()));
                $montantTotal += (float)$ligne->getMontantTotal();
                
                $lignes[] = ['ligne' => $ligne, 'produit' => $produit];
            }

            $reduction = (float)$vente->getReduction();
            $vente->setMontantTotal((string)$montantTotal);
            $vente->setMontantFinal((string)($montantTotal - $reduction));
            
            // Persister la vente d'abord
            $this->em->persist($vente);
            $this->em->flush();
            
            // Ensuite créer les lignes et les mouvements
            foreach ($lignes as $item) {
                $this->em->persist($item['ligne']);
                
                // Mise à jour du stock
                $this->stockService->enregistrerMouvement(
                    $item['produit'],
                    \App\Entity\MouvementStock::TYPE_SORTIE_VENTE,
                    $item['ligne']->getQuantite(),
                    $vente->getVendeur(),
                    "Vente #{$vente->getNumeroFacture()}",
                    $vente
                );
            }
        }
        $this->em->flush();
        $io->info('✓ 50 ventes créées');
    }

    private function createDepenses(Magasin $magasin, SymfonyStyle $io): void
    {
        $users = $this->em->getRepository(User::class)->findBy(['magasin' => $magasin]);
        if (empty($users)) {
            $io->warning('Aucun utilisateur trouvé pour créer les dépenses');
            return;
        }

        $categories = [
            Depense::CATEGORIE_ACHAT_STOCK,
            Depense::CATEGORIE_LOYER,
            Depense::CATEGORIE_SALAIRES,
            Depense::CATEGORIE_TRANSPORT,
            Depense::CATEGORIE_DIVERS
        ];

        $libelles = [
            'Achat stock mensuel', 'Loyer magasin', 'Salaire employé', 'Transport marchandises',
            'Frais divers', 'Maintenance équipement', 'Publicité', 'Assurance'
        ];

        for ($i = 0; $i < 30; $i++) {
            $depense = new Depense();
            $depense->setLibelle($libelles[array_rand($libelles)]);
            $depense->setCategorie($categories[array_rand($categories)]);
            $depense->setMontant((string)rand(50, 2000));
            $depense->setMagasin($magasin);
            $depense->setUtilisateur($users[array_rand($users)]);
            $depense->setDateDepense(new \DateTime('-' . rand(1, 60) . ' days'));
            $this->em->persist($depense);
        }
        $this->em->flush();
        $io->info('✓ 30 dépenses créées');
    }
}

