<?php

namespace App\Controller;

use App\Service\DashboardService;
use App\Repository\MagasinRepository;
use App\Repository\UserRepository;
use App\Repository\CategorieRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/', name: 'dashboard')]
#[IsGranted('ROLE_USER')]
class DashboardController extends AbstractController
{
    #[Route('', name: '_index')]
    public function index(
        DashboardService $dashboardService,
        MagasinRepository $magasinRepository,
        UserRepository $userRepository,
        CategorieRepository $categorieRepository,
        Request $request
    ): Response {
        $user = $this->getUser();
        
        // Récupérer les filtres
        $period = $request->query->get('period', 'this_month');
        $magasinId = $request->query->get('magasin');
        $userId = $request->query->get('user');
        $categorieId = $request->query->get('categorie');
        $dateDebutCustom = $request->query->get('date_debut') ? new \DateTime($request->query->get('date_debut')) : null;
        $dateFinCustom = $request->query->get('date_fin') ? new \DateTime($request->query->get('date_fin')) : null;
        
        // Calculer les dates selon la période
        [$dateDebut, $dateFin] = $dashboardService->getDatesFromPeriod($period, $dateDebutCustom, $dateFinCustom);
        
        // Récupérer les magasins selon le rôle
        $magasinsDisponibles = $dashboardService->getMagasinsForUser($user);
        $magasins = $magasinId 
            ? array_filter($magasinsDisponibles, fn($m) => $m->getId() == $magasinId)
            : $magasinsDisponibles;
        
        // Récupérer l'utilisateur si filtré
        $userFilter = $userId ? $userRepository->find($userId) : null;
        
        // Récupérer la catégorie si filtrée
        $categorieFilter = $categorieId ? $categorieRepository->find($categorieId) : null;
        
        // Calculer les KPIs
        $kpis = $dashboardService->getKPIs(
            $magasins,
            $dateDebut,
            $dateFin,
            $userFilter,
            $categorieId
        );
        
        // Récupérer les alertes
        $alertes = $dashboardService->getAlertes($magasins);
        
        // Récupérer les données pour les graphiques
        $graphiques = $dashboardService->getGraphiques($magasins, $dateDebut, $dateFin, $userFilter);
        
        // Récupérer les utilisateurs pour le filtre
        $users = [];
        foreach ($magasins as $magasin) {
            $users = array_merge($users, $magasin->getUsers()->filter(fn($u) => $u->isActif())->toArray());
        }
        
        // Récupérer les catégories pour le filtre
        $categories = [];
        foreach ($magasins as $magasin) {
            $categories = array_merge($categories, $magasin->getCategories()->toArray());
        }
        
        return $this->render('dashboard/index.html.twig', [
            'kpis' => $kpis,
            'alertes' => $alertes,
            'graphiques' => $graphiques,
            'magasins' => $magasinsDisponibles,
            'users' => $users,
            'categories' => $categories,
            'filters' => [
                'period' => $period,
                'magasin' => $magasinId,
                'user' => $userId,
                'categorie' => $categorieId,
                'date_debut' => $dateDebutCustom ? $dateDebutCustom->format('Y-m-d') : null,
                'date_fin' => $dateFinCustom ? $dateFinCustom->format('Y-m-d') : null,
            ],
            'dateDebut' => $dateDebut,
            'dateFin' => $dateFin,
        ]);
    }
}
