<?php

namespace App\Controller;

use App\Entity\Fournisseur;
use App\Form\FournisseurType;
use App\Repository\FournisseurRepository;
use App\Service\LogService;
use App\Service\FournisseurService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/fournisseurs', name: 'fournisseurs_')]
#[IsGranted('ROLE_USER')]
class FournisseurController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(FournisseurRepository $fournisseurRepository, FournisseurService $fournisseurService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $fournisseurs = $fournisseurRepository->findBy(
            ['magasin' => $magasin],
            ['nom' => 'ASC']
        );

        // Calculer les totaux pour chaque fournisseur
        $fournisseursData = [];
        foreach ($fournisseurs as $fournisseur) {
            $fournisseursData[] = [
                'fournisseur' => $fournisseur,
                'totalAchats' => $fournisseurService->getTotalAchats($fournisseur),
                'totalPaye' => $fournisseurService->getTotalPaye($fournisseur),
                'resteDu' => $fournisseurService->getResteDu($fournisseur),
            ];
        }

        return $this->render('fournisseur/index.html.twig', [
            'fournisseurs' => $fournisseurs,
            'fournisseursData' => $fournisseursData,
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function new(Request $request, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $fournisseur = new Fournisseur();
        $fournisseur->setMagasin($magasin);
        $fournisseur->setStatut(Fournisseur::STATUT_ACTIF);

        $form = $this->createForm(FournisseurType::class, $fournisseur);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $fournisseur->setDateModification(new \DateTime());
            $em->persist($fournisseur);
            $em->flush();

            $logService->log($user, $magasin, 'Création fournisseur', 'Fournisseur', $fournisseur->getId(), "Fournisseur: {$fournisseur->getNom()}");

            $this->addFlash('success', 'Fournisseur créé avec succès');
            return $this->redirectToRoute('fournisseurs_show', ['id' => $fournisseur->getId()]);
        }

        return $this->render('fournisseur/new.html.twig', [
            'fournisseur' => $fournisseur,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Fournisseur $fournisseur, FournisseurService $fournisseurService): Response
    {
        $user = $this->getUser();
        if ($fournisseur->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        // Récupérer les statistiques
        $totalAchats = $fournisseurService->getTotalAchats($fournisseur);
        $totalPaye = $fournisseurService->getTotalPaye($fournisseur);
        $resteDu = $fournisseurService->getResteDu($fournisseur);
        $dernierAchat = $fournisseurService->getDernierAchat($fournisseur);
        $prochainPaiement = $fournisseurService->getProchainPaiement($fournisseur);
        $paiementsEnRetard = $fournisseurService->getPaiementsEnRetard($fournisseur);
        $articlesPlusAchetes = $fournisseurService->getArticlesPlusAchetes($fournisseur);
        $totalArticlesAchetes = $fournisseurService->getTotalArticlesAchetes($fournisseur);

        // Historique des opérations (achats et paiements triés par date)
        $historique = [];
        foreach ($fournisseur->getAchats() as $achat) {
            $historique[] = [
                'type' => 'achat',
                'date' => $achat->getDateAchat(),
                'achat' => $achat,
                'description' => sprintf(
                    'Achat : %s - %s MAD (Reste: %s MAD)',
                    $achat->getNumeroFacture(),
                    number_format((float)$achat->getMontantTotal(), 2, ',', ' '),
                    number_format((float)$achat->getMontantRestant(), 2, ',', ' ')
                )
            ];
        }
        foreach ($fournisseur->getPaiements() as $paiement) {
            $historique[] = [
                'type' => 'paiement',
                'date' => $paiement->getDatePaiement(),
                'paiement' => $paiement,
                'description' => sprintf(
                    'Paiement : %s MAD (%s)',
                    number_format((float)$paiement->getMontant(), 2, ',', ' '),
                    $paiement->getMode()
                )
            ];
        }

        // Trier par date décroissante
        usort($historique, function($a, $b) {
            return $b['date'] <=> $a['date'];
        });

        return $this->render('fournisseur/show.html.twig', [
            'fournisseur' => $fournisseur,
            'totalAchats' => $totalAchats,
            'totalPaye' => $totalPaye,
            'resteDu' => $resteDu,
            'dernierAchat' => $dernierAchat,
            'prochainPaiement' => $prochainPaiement,
            'paiementsEnRetard' => $paiementsEnRetard,
            'articlesPlusAchetes' => $articlesPlusAchetes,
            'totalArticlesAchetes' => $totalArticlesAchetes,
            'historique' => $historique,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function edit(Request $request, Fournisseur $fournisseur, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        if ($fournisseur->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(FournisseurType::class, $fournisseur);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $fournisseur->setDateModification(new \DateTime());
            $em->flush();

            $logService->log($user, $fournisseur->getMagasin(), 'Modification fournisseur', 'Fournisseur', $fournisseur->getId(), "Fournisseur: {$fournisseur->getNom()}");

            $this->addFlash('success', 'Fournisseur modifié avec succès');
            return $this->redirectToRoute('fournisseurs_show', ['id' => $fournisseur->getId()]);
        }

        return $this->render('fournisseur/edit.html.twig', [
            'fournisseur' => $fournisseur,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/desactivate', name: 'desactivate', methods: ['POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function desactivate(Request $request, Fournisseur $fournisseur, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        if ($fournisseur->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('desactivate' . $fournisseur->getId(), $request->request->get('_token'))) {
            $fournisseur->setStatut($fournisseur->isActif() ? Fournisseur::STATUT_DESACTIVE : Fournisseur::STATUT_ACTIF);
            $fournisseur->setDateModification(new \DateTime());
            $em->flush();

            $action = $fournisseur->isActif() ? 'Réactivation' : 'Désactivation';
            $logService->log($user, $fournisseur->getMagasin(), $action . ' fournisseur', 'Fournisseur', $fournisseur->getId(), "Fournisseur: {$fournisseur->getNom()}");

            $this->addFlash('success', $fournisseur->isActif() ? 'Fournisseur réactivé' : 'Fournisseur désactivé');
        }

        return $this->redirectToRoute('fournisseurs_show', ['id' => $fournisseur->getId()]);
    }

    #[Route('/alertes', name: 'alertes', methods: ['GET'])]
    public function alertes(FournisseurRepository $fournisseurRepository, FournisseurService $fournisseurService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $fournisseurs = $fournisseurRepository->findBy(['magasin' => $magasin]);

        $alertes = [];
        foreach ($fournisseurs as $fournisseur) {
            $paiementsEnRetard = $fournisseurService->getPaiementsEnRetard($fournisseur);
            $prochainPaiement = $fournisseurService->getProchainPaiement($fournisseur);

            foreach ($paiementsEnRetard as $achat) {
                $alertes[] = [
                    'type' => 'retard',
                    'fournisseur' => $fournisseur,
                    'achat' => $achat,
                    'date' => $achat->getDateLimitePaiement(),
                    'montant' => (float)$achat->getMontantRestant(),
                ];
            }

            if ($prochainPaiement) {
                $dateLimite = $prochainPaiement->getDateLimitePaiement();
                $aujourdhui = new \DateTime();
                $joursRestants = $aujourdhui->diff($dateLimite)->days;

                if ($joursRestants <= 7) {
                    $alertes[] = [
                        'type' => 'a_venir',
                        'fournisseur' => $fournisseur,
                        'achat' => $prochainPaiement,
                        'date' => $dateLimite,
                        'montant' => (float)$prochainPaiement->getMontantRestant(),
                        'joursRestants' => $joursRestants,
                    ];
                }
            }
        }

        // Trier par date
        usort($alertes, function($a, $b) {
            return $a['date'] <=> $b['date'];
        });

        return $this->render('fournisseur/alertes.html.twig', [
            'alertes' => $alertes,
        ]);
    }
}
