<?php

namespace App\Controller;

use App\Entity\Produit;
use App\Entity\Categorie;
use App\Entity\SousCategorie;
use App\Entity\Achat;
use App\Entity\AchatParticulier;
use App\Form\ProduitType;
use App\Repository\ProduitRepository;
use App\Repository\CategorieRepository;
use App\Service\CodeBarresService;
use App\Service\LogService;
use App\Service\StockService;
use App\Service\FactureService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/produits', name: 'produits_')]
#[IsGranted('ROLE_USER')]
class ProduitController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(
        ProduitRepository $produitRepository,
        CategorieRepository $categorieRepository,
        EntityManagerInterface $em,
        Request $request
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        // Récupérer tous les paramètres de filtres
        $search = $request->query->get('search', '');
        $categorieId = $request->query->get('categorie') ? (int)$request->query->get('categorie') : null;
        $sousCategorieId = $request->query->get('sous_categorie') ? (int)$request->query->get('sous_categorie') : null;
        $codeBarres = $request->query->get('code_barres');
        $referenceInterne = $request->query->get('reference_interne');
        $statut = $request->query->get('statut');
        $fournisseurId = $request->query->get('fournisseur') ? (int)$request->query->get('fournisseur') : null;
        $prixMin = $request->query->get('prix_min') ? (float)$request->query->get('prix_min') : null;
        $prixMax = $request->query->get('prix_max') ? (float)$request->query->get('prix_max') : null;
        $imei = $request->query->get('imei');
        $numeroSerie = $request->query->get('numero_serie');
        
        // Récupérer les filtres de caractéristiques
        $caracteristiques = [];
        foreach ($request->query->all() as $key => $value) {
            if (strpos($key, 'caracteristique_') === 0 && $value) {
                $caracId = str_replace('caracteristique_', '', $key);
                $caracteristiques[$caracId] = $value;
            }
        }

        // Récupérer le paramètre de tri (par défaut: date_entree_desc = plus récents)
        $orderBy = $request->query->get('order_by', 'date_entree_desc');

        $produits = $produitRepository->findByMagasin(
            $magasin,
            $search,
            $categorieId,
            $sousCategorieId,
            $codeBarres,
            $referenceInterne,
            $statut,
            $fournisseurId,
            $prixMin,
            $prixMax,
            $imei,
            $numeroSerie,
            $caracteristiques,
            $orderBy
        );

        // Récupérer les catégories et sous-catégories pour les filtres
        $categories = $categorieRepository->findBy(['magasin' => $magasin], ['nom' => 'ASC']);
        $sousCategories = [];
        if ($categorieId) {
            $categorie = $em->getRepository(Categorie::class)->find($categorieId);
            if ($categorie) {
                $sousCategories = $categorie->getSousCategories()->toArray();
            }
        }

        // Récupérer les fournisseurs
        $fournisseurs = $magasin->getFournisseurs()->filter(fn($f) => $f->isActif())->toArray();

        // Récupérer les caractéristiques disponibles si une catégorie/sous-catégorie est sélectionnée
        $caracteristiquesDisponibles = [];
        if ($sousCategorieId) {
            $sousCategorie = $em->getRepository(SousCategorie::class)->find($sousCategorieId);
            if ($sousCategorie) {
                // Caractéristiques de la catégorie parente
                $categorie = $sousCategorie->getCategorie();
                if ($categorie) {
                    foreach ($categorie->getCaracteristiques() as $carac) {
                        $caracteristiquesDisponibles[] = $carac;
                    }
                }
                // Caractéristiques de la sous-catégorie
                foreach ($sousCategorie->getCaracteristiques() as $carac) {
                    $caracteristiquesDisponibles[] = $carac;
                }
            }
        } elseif ($categorieId) {
            $categorie = $em->getRepository(Categorie::class)->find($categorieId);
            if ($categorie) {
                foreach ($categorie->getCaracteristiques() as $carac) {
                    $caracteristiquesDisponibles[] = $carac;
                }
            }
        }

        // Option pour afficher les produits groupés ou individuels
        $vueGroupee = $request->query->get('vue') === 'groupee';
        $produitsGroupes = [];
        
        if ($vueGroupee) {
            $produitsGroupes = $produitRepository->getProduitsGroupes($magasin);
        }

        return $this->render('produit/index.html.twig', [
            'produits' => $produits,
            'produitsGroupes' => $produitsGroupes,
            'vueGroupee' => $vueGroupee,
            'categories' => $categories,
            'sousCategories' => $sousCategories,
            'fournisseurs' => $fournisseurs,
            'caracteristiquesDisponibles' => $caracteristiquesDisponibles,
            'filters' => [
                'search' => $search ?? '',
                'categorie' => $categorieId,
                'sous_categorie' => $sousCategorieId,
                'code_barres' => $codeBarres ?? '',
                'reference_interne' => $referenceInterne ?? '',
                'statut' => $statut ?? '',
                'fournisseur' => $fournisseurId,
                'prix_min' => $prixMin,
                'prix_max' => $prixMax,
                'imei' => $imei ?? '',
                'numero_serie' => $numeroSerie ?? '',
                'caracteristiques' => $caracteristiques ?? [],
                'order_by' => $orderBy,
            ],
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function new(
        Request $request,
        EntityManagerInterface $em,
        CodeBarresService $codeBarresService,
        LogService $logService,
        FactureService $factureService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $produit = new Produit();
        $produit->setMagasin($magasin);
        // La quantité est toujours 1 (chaque produit est individuel)
        $produit->setQuantite(1);
        $codeBarres = $codeBarresService->genererCodeBarres();
        $produit->setCodeBarres($codeBarres);
        // Générer automatiquement la référence interne basée sur le code-barres
        $produit->setReferenceInterne($codeBarresService->genererReferenceInterne($codeBarres));

        $form = $this->createForm(ProduitType::class, $produit, ['magasin' => $magasin]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Traiter les caractéristiques dynamiques
            $caracteristiquesValeurs = [];
            foreach ($request->request->all() as $key => $value) {
                if (strpos($key, 'caracteristique_') === 0) {
                    $caracteristiqueId = str_replace('caracteristique_', '', $key);
                    $caracteristiqueId = str_replace('[]', '', $caracteristiqueId);
                    $caracteristiquesValeurs[$caracteristiqueId] = $value;
                }
            }
            
            // Si les valeurs sont dans le champ caché JSON
            $caracteristiquesJson = $request->request->get('caracteristiques_valeurs');
            if ($caracteristiquesJson) {
                $caracteristiquesValeurs = array_merge($caracteristiquesValeurs, json_decode($caracteristiquesJson, true) ?? []);
            }
            
            $produit->setCaracteristiquesValeurs($caracteristiquesValeurs);
            
            // Calculer le prix de vente recommandé
            if ($produit->getPrixAchat() && $produit->getMargeMinimale()) {
                $produit->calculerPrixVenteRecommande();
            }
            
            // Générer le nom automatiquement à partir de la sous-catégorie et des caractéristiques
            if ($produit->getSousCategorie()) {
                $nom = $produit->getSousCategorie()->getNom();
                if (!empty($caracteristiquesValeurs)) {
                    $valeurs = [];
                    foreach ($caracteristiquesValeurs as $caracId => $valeur) {
                        if (is_array($valeur)) {
                            // Pour les multi-select, prendre toutes les valeurs
                            $valeurs = array_merge($valeurs, array_filter($valeur));
                        } else {
                            // Pour les autres types, prendre la valeur si elle n'est pas vide
                            if ($valeur !== null && $valeur !== '' && $valeur !== '0' && $valeur !== 0) {
                                $valeurs[] = $valeur;
                            }
                        }
                    }
                    if (!empty($valeurs)) {
                        $nom .= ' - ' . implode(' / ', array_slice($valeurs, 0, 5)); // Limiter à 5 valeurs max
                    }
                }
                $produit->setNom($nom);
            }
            
            // Récupérer la quantité demandée
            $quantite = (int)($request->request->get('quantite', 1));
            if ($quantite < 1) {
                $quantite = 1;
            }
            
            // Si quantité > 1, créer plusieurs produits distincts
            $produitsCrees = [];
            $codesBarresUtilises = []; // Pour éviter les doublons dans la même transaction
            
            for ($i = 0; $i < $quantite; $i++) {
                if ($i === 0) {
                    // Premier produit : utiliser celui du formulaire
                    $produitCourant = $produit;
                    // Ajouter son code-barres à la liste
                    if ($produit->getCodeBarres()) {
                        $codesBarresUtilises[] = $produit->getCodeBarres();
                    }
                } else {
                    // Produits suivants : cloner le premier
                    $produitCourant = clone $produit;
                    // Réinitialiser l'ID pour que Doctrine crée un nouvel enregistrement
                    $reflection = new \ReflectionClass($produitCourant);
                    $idProperty = $reflection->getProperty('id');
                    $idProperty->setAccessible(true);
                    $idProperty->setValue($produitCourant, null);
                    
                    // Générer un nouveau code-barres unique (en excluant ceux déjà utilisés)
                    $codeBarres = $codeBarresService->genererCodeBarres($codesBarresUtilises);
                    $produitCourant->setCodeBarres($codeBarres);
                    $produitCourant->setReferenceInterne($codeBarresService->genererReferenceInterne($codeBarres));
                    
                    // Ajouter le nouveau code-barres à la liste
                    $codesBarresUtilises[] = $codeBarres;
                }
                
                $em->persist($produitCourant);
                $produitsCrees[] = $produitCourant;
            }
            
            $em->flush();
            
            // Utiliser le premier produit créé pour les relations (achat particulier, etc.)
            $produit = $produitsCrees[0];
            
            // Gérer l'achat particulier si l'origine est "particulier"
            if ($produit->getOrigineAchat() === Produit::ORIGINE_PARTICULIER) {
                $nomClient = $request->request->get('nom_client');
                $telephoneClient = $request->request->get('telephone_client');
                $cinClient = $request->request->get('cin_client');
                $adresseClient = $request->request->get('adresse_client');
                $prixPayeParticulier = $request->request->get('prix_paye_particulier');
                
                if ($nomClient && $telephoneClient && $prixPayeParticulier) {
                    $achatParticulier = new AchatParticulier();
                    $achatParticulier->setProduit($produit);
                    $achatParticulier->setUtilisateur($user);
                    $achatParticulier->setMagasin($magasin);
                    $achatParticulier->setNomClient($nomClient);
                    $achatParticulier->setTelephone($telephoneClient);
                    $achatParticulier->setCin($cinClient);
                    $achatParticulier->setAdresse($adresseClient);
                    $achatParticulier->setPrixPaye($prixPayeParticulier);
                    $achatParticulier->setDateAchat($produit->getDateAchat() ?? new \DateTime());
                    
                    // Gérer l'upload de la photo CIN si présente
                    $photoCinFile = $request->files->get('photo_cin');
                    if ($photoCinFile) {
                        $uploadDir = $this->getParameter('kernel.project_dir') . '/public/uploads/cin/';
                        if (!is_dir($uploadDir)) {
                            mkdir($uploadDir, 0755, true);
                        }
                        
                        $filename = uniqid() . '.' . $photoCinFile->guessExtension();
                        $photoCinFile->move($uploadDir, $filename);
                        $achatParticulier->setPhotoCin('/uploads/cin/' . $filename);
                    }
                    
                    $produit->setAchatParticulier($achatParticulier);
                    $em->persist($achatParticulier);
                    $em->flush();
                }
            }

            // Gérer l'achat fournisseur et paiement si l'origine est "fournisseur"
            if ($produit->getOrigineAchat() === Produit::ORIGINE_FOURNISSEUR && $produit->getFournisseur()) {
                $creerAchat = $request->request->get('creer_achat', false);
                $montantPaiement = $request->request->get('montant_paiement');
                $modePaiement = $request->request->get('mode_paiement', \App\Entity\Paiement::MODE_ESPECES);
                $referencePaiement = $request->request->get('reference_paiement');
                $notesPaiement = $request->request->get('notes_paiement');
                
                if ($creerAchat) {
                    // Créer un achat pour ce produit
                    $achat = new Achat();
                    $achat->setFournisseur($produit->getFournisseur());
                    $achat->setMagasin($magasin);
                    $achat->setUtilisateur($user);
                    $achat->setNumeroFacture($factureService->genererNumeroFactureAchat());
                    $achat->setDateAchat($produit->getDateAchat() ?? new \DateTime());
                    
                    // Calculer le montant total (prix d'achat * quantité de produits créés)
                    $montantTotal = (float)$produit->getPrixAchat() * $quantite;
                    $achat->setMontantTotal((string)$montantTotal);
                    $achat->setMontantRestant((string)$montantTotal);
                    
                    // Créer une ligne d'achat pour chaque produit
                    foreach ($produitsCrees as $prod) {
                        $ligneAchat = new \App\Entity\LigneAchat();
                        $ligneAchat->setAchat($achat);
                        $ligneAchat->setProduit($prod);
                        $ligneAchat->setNomProduit($prod->getNom());
                        $ligneAchat->setQuantite(1);
                        $ligneAchat->setPrixUnitaire($prod->getPrixAchat());
                        $ligneAchat->setMontantTotal($prod->getPrixAchat());
                        $achat->addLigneAchat($ligneAchat);
                        $prod->setAchatFournisseur($achat);
                    }
                    
                    $em->persist($achat);
                    
                    // Si un paiement est spécifié, l'enregistrer
                    if ($montantPaiement && (float)$montantPaiement > 0) {
                        $paiement = new \App\Entity\Paiement();
                        $paiement->setFournisseur($produit->getFournisseur());
                        $paiement->setAchat($achat);
                        $paiement->setUtilisateur($user);
                        $paiement->setMontant((string)$montantPaiement);
                        $paiement->setMode($modePaiement);
                        $paiement->setReference($referencePaiement);
                        $paiement->setNotes($notesPaiement);
                        $paiement->setDatePaiement(new \DateTime());
                        
                        // Ajouter le paiement à l'achat pour la relation bidirectionnelle
                        $achat->addPaiement($paiement);
                        
                        $em->persist($paiement);
                        
                        // Mettre à jour le montant restant après le paiement
                        $montantPaye = (float)$montantPaiement;
                        $nouveauMontantRestant = max(0, $montantTotal - $montantPaye);
                        $achat->setMontantRestant((string)$nouveauMontantRestant);
                        
                        $logService->log(
                            $user,
                            $magasin,
                            'Paiement fournisseur',
                            'Paiement',
                            $paiement->getId(),
                            "Paiement de {$montantPaiement} MAD pour l'achat #{$achat->getNumeroFacture()}"
                        );
                    }
                    
                    $em->flush();
                }
            }

            $logService->log($user, $magasin, 'Ajout produit', 'Produit', $produit->getId(), "Produit: {$produit->getNom()}");

            return $this->redirectToRoute('produits_index');
        }

        return $this->render('produit/new.html.twig', [
            'produit' => $produit,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Produit $produit): Response
    {
        $user = $this->getUser();
        if ($produit->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('produit/show.html.twig', [
            'produit' => $produit,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function edit(
        Request $request,
        Produit $produit,
        EntityManagerInterface $em,
        LogService $logService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($produit->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(ProduitType::class, $produit, ['magasin' => $magasin]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->flush();

            $logService->log($user, $magasin, 'Modification produit', 'Produit', $produit->getId(), "Produit: {$produit->getNom()}");

            return $this->redirectToRoute('produits_index');
        }

        return $this->render('produit/edit.html.twig', [
            'produit' => $produit,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'delete', methods: ['POST'])]
    #[IsGranted('ROLE_MANAGER')]
    public function delete(
        Request $request,
        Produit $produit,
        EntityManagerInterface $em,
        LogService $logService
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($produit->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $produit->getId(), $request->request->get('_token'))) {
            $produit->setActif(false);
            $em->flush();

            $logService->log($user, $magasin, 'Suppression produit', 'Produit', $produit->getId(), "Produit: {$produit->getNom()}");
        }

        return $this->redirectToRoute('produits_index');
    }

    #[Route('/api/sous-categories/{categorieId}', name: 'api_sous_categories', methods: ['GET'])]
    public function getSousCategories(int $categorieId, EntityManagerInterface $em): Response
    {
        try {
            $user = $this->getUser();
            if (!$user) {
                return $this->json(['error' => 'Non authentifié'], 401);
            }

            $magasin = $user->getMagasin();
            if (!$magasin) {
                return $this->json(['error' => 'Magasin non trouvé'], 404);
            }

            $categorie = $em->getRepository(Categorie::class)->find($categorieId);
            if (!$categorie || $categorie->getMagasin() !== $magasin) {
                return $this->json(['error' => 'Catégorie non trouvée'], 404);
            }

            $sousCategories = [];
            foreach ($categorie->getSousCategories() as $sousCategorie) {
                $sousCategories[] = [
                    'id' => $sousCategorie->getId(),
                    'nom' => $sousCategorie->getNom(),
                ];
            }

            return $this->json($sousCategories);
        } catch (\Exception $e) {
            return $this->json(['error' => 'Erreur serveur: ' . $e->getMessage()], 500);
        }
    }

    #[Route('/api/caracteristiques/{sousCategorieId}', name: 'api_caracteristiques', methods: ['GET'])]
    public function getCaracteristiques(int $sousCategorieId, EntityManagerInterface $em): Response
    {
        try {
            $user = $this->getUser();
            if (!$user) {
                return $this->json(['error' => 'Non authentifié'], 401);
            }

            $magasin = $user->getMagasin();
            if (!$magasin) {
                return $this->json(['error' => 'Magasin non trouvé'], 404);
            }

            $sousCategorie = $em->getRepository(SousCategorie::class)->find($sousCategorieId);
            if (!$sousCategorie || $sousCategorie->getMagasin() !== $magasin) {
                return $this->json(['error' => 'Sous-catégorie non trouvée'], 404);
            }

            $caracteristiques = [];
            
            // Caractéristiques de la catégorie parente
            $categorie = $sousCategorie->getCategorie();
            if ($categorie) {
                foreach ($categorie->getCaracteristiques() as $caracteristique) {
                    $caracteristiques[] = [
                        'id' => $caracteristique->getId(),
                        'nom' => $caracteristique->getNom(),
                        'type' => $caracteristique->getType(),
                        'valeursPossibles' => $caracteristique->getValeursPossibles(),
                        'obligatoire' => $caracteristique->isObligatoire(),
                    ];
                }
            }

            // Caractéristiques de la sous-catégorie
            foreach ($sousCategorie->getCaracteristiques() as $caracteristique) {
                $caracteristiques[] = [
                    'id' => $caracteristique->getId(),
                    'nom' => $caracteristique->getNom(),
                    'type' => $caracteristique->getType(),
                    'valeursPossibles' => $caracteristique->getValeursPossibles(),
                    'obligatoire' => $caracteristique->isObligatoire(),
                ];
            }

            return $this->json($caracteristiques);
        } catch (\Exception $e) {
            return $this->json(['error' => 'Erreur serveur: ' . $e->getMessage()], 500);
        }
    }

    #[Route('/{id}/imprimer-code-barres', name: 'imprimer_code_barres', methods: ['GET'])]
    public function imprimerCodeBarres(Produit $produit): Response
    {
        $user = $this->getUser();
        if ($produit->getMagasin() !== $user->getMagasin()) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('produit/imprimer_code_barres.html.twig', [
            'produit' => $produit,
        ]);
    }
}

