<?php

namespace App\Entity;

use App\Repository\ProduitRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity(repositoryClass: ProduitRepository::class)]
#[ORM\Table(name: 'produits')]
class Produit
{
    const STATUT_EN_STOCK = 'en_stock';
    const STATUT_RESERVE = 'reserve';
    const STATUT_VENDU = 'vendu';
    const STATUT_EN_REPARATION = 'en_reparation';
    const STATUT_RETOURNE = 'retourne';
    const STATUT_PERDU = 'perdu';
    const STATUT_TRANSFERE = 'transfere';

    const ORIGINE_FOURNISSEUR = 'fournisseur';
    const ORIGINE_PARTICULIER = 'particulier';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255)]
    private ?string $nom = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $referenceInterne = null;

    #[ORM\Column(length: 50, nullable: true, unique: true)]
    private ?string $imei = null; // Obligatoire pour téléphones

    #[ORM\Column(length: 100, nullable: true)]
    private ?string $numeroSerie = null;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $description = null;

    #[ORM\ManyToOne(inversedBy: 'produits')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Categorie $categorie = null;

    #[ORM\ManyToOne(inversedBy: 'produits')]
    #[ORM\JoinColumn(nullable: false)] // Obligatoire maintenant
    private ?SousCategorie $sousCategorie = null;

    #[ORM\ManyToOne(inversedBy: 'produits')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Magasin $magasin = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2)]
    private ?string $prixAchat = null;

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
    private ?string $margeMinimale = null;

    #[ORM\Column(name: 'prix_vente_recommande', type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
    private ?string $prixVenteRecommande = null; // Calculé automatiquement

    #[ORM\Column(type: Types::DECIMAL, precision: 10, scale: 2, nullable: true)]
    private ?string $prixVente = null; // Prix de vente final si vendu

    #[ORM\Column(type: Types::INTEGER)]
    private int $quantite = 1; // Toujours 1 (pas de lots)

    #[ORM\Column(length: 100, unique: true)]
    private ?string $codeBarres = null;

    #[ORM\Column(length: 50)]
    private ?string $statut = self::STATUT_EN_STOCK;

    #[ORM\Column(length: 50)]
    private ?string $origineAchat = self::ORIGINE_FOURNISSEUR;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateAchat = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateEntree = null;

    #[ORM\ManyToOne(inversedBy: 'produits')]
    private ?Fournisseur $fournisseur = null;

    #[ORM\ManyToOne]
    private ?Achat $achatFournisseur = null; // Lien vers l'achat si acheté via fournisseur

    #[ORM\Column(type: Types::JSON, nullable: true)]
    private ?array $caracteristiquesValeurs = [];

    #[ORM\Column(type: Types::BOOLEAN)]
    private bool $actif = true;

    #[ORM\OneToMany(mappedBy: 'produit', targetEntity: ImageProduit::class, cascade: ['persist', 'remove'])]
    private Collection $images;

    #[ORM\OneToOne(mappedBy: 'produit', targetEntity: AchatParticulier::class, cascade: ['persist', 'remove'])]
    private ?AchatParticulier $achatParticulier = null;

    public function __construct()
    {
        $this->images = new ArrayCollection();
        $this->dateEntree = new \DateTime();
        $this->dateAchat = new \DateTime();
        $this->caracteristiquesValeurs = [];
        $this->statut = self::STATUT_EN_STOCK;
        $this->origineAchat = self::ORIGINE_FOURNISSEUR;
        $this->quantite = 1;
    }

    /**
     * @return Collection<int, ImageProduit>
     */
    public function getImages(): Collection
    {
        return $this->images;
    }

    public function addImage(ImageProduit $image): self
    {
        if (!$this->images->contains($image)) {
            $this->images->add($image);
            $image->setProduit($this);
        }
        return $this;
    }

    public function removeImage(ImageProduit $image): self
    {
        if ($this->images->removeElement($image)) {
            if ($image->getProduit() === $this) {
                $image->setProduit(null);
            }
        }
        return $this;
    }

    public function getAchatParticulier(): ?AchatParticulier
    {
        return $this->achatParticulier;
    }

    public function setAchatParticulier(?AchatParticulier $achatParticulier): self
    {
        if ($achatParticulier && $achatParticulier->getProduit() !== $this) {
            $achatParticulier->setProduit($this);
        }
        $this->achatParticulier = $achatParticulier;
        return $this;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getNom(): ?string
    {
        return $this->nom;
    }

    public function setNom(string $nom): self
    {
        $this->nom = $nom;
        return $this;
    }

    public function getDescription(): ?string
    {
        return $this->description;
    }

    public function setDescription(?string $description): self
    {
        $this->description = $description;
        return $this;
    }

    public function getCategorie(): ?Categorie
    {
        return $this->categorie;
    }

    public function setCategorie(?Categorie $categorie): self
    {
        $this->categorie = $categorie;
        return $this;
    }

    public function getMagasin(): ?Magasin
    {
        return $this->magasin;
    }

    public function setMagasin(?Magasin $magasin): self
    {
        $this->magasin = $magasin;
        return $this;
    }

    public function getPrixAchat(): ?string
    {
        return $this->prixAchat;
    }

    public function setPrixAchat(string $prixAchat): self
    {
        $this->prixAchat = $prixAchat;
        return $this;
    }

    public function getPrixVente(): ?string
    {
        return $this->prixVente;
    }

    public function setPrixVente(string $prixVente): self
    {
        $this->prixVente = $prixVente;
        return $this;
    }

    public function getQuantite(): int
    {
        return $this->quantite;
    }

    public function setQuantite(int $quantite): self
    {
        $this->quantite = $quantite;
        return $this;
    }

    public function getCodeBarres(): ?string
    {
        return $this->codeBarres;
    }

    public function setCodeBarres(string $codeBarres): self
    {
        $this->codeBarres = $codeBarres;
        return $this;
    }

    public function getReferenceInterne(): ?string
    {
        return $this->referenceInterne;
    }

    public function setReferenceInterne(?string $referenceInterne): self
    {
        $this->referenceInterne = $referenceInterne;
        return $this;
    }

    public function getImei(): ?string
    {
        return $this->imei;
    }

    public function setImei(?string $imei): self
    {
        $this->imei = $imei;
        return $this;
    }

    public function getNumeroSerie(): ?string
    {
        return $this->numeroSerie;
    }

    public function setNumeroSerie(?string $numeroSerie): self
    {
        $this->numeroSerie = $numeroSerie;
        return $this;
    }

    public function getMargeMinimale(): ?string
    {
        return $this->margeMinimale;
    }

    public function setMargeMinimale(?string $margeMinimale): self
    {
        $this->margeMinimale = $margeMinimale;
        return $this;
    }

    public function getPrixVenteRecommande(): ?string
    {
        return $this->prixVenteRecommande;
    }

    public function setPrixVenteRecommande(?string $prixVenteRecommande): self
    {
        $this->prixVenteRecommande = $prixVenteRecommande;
        return $this;
    }

    public function getStatut(): ?string
    {
        return $this->statut;
    }

    public function setStatut(string $statut): self
    {
        $this->statut = $statut;
        return $this;
    }

    public function getOrigineAchat(): ?string
    {
        return $this->origineAchat;
    }

    public function setOrigineAchat(string $origineAchat): self
    {
        $this->origineAchat = $origineAchat;
        return $this;
    }

    public function getDateAchat(): ?\DateTimeInterface
    {
        return $this->dateAchat;
    }

    public function setDateAchat(?\DateTimeInterface $dateAchat): self
    {
        $this->dateAchat = $dateAchat;
        return $this;
    }

    public function getAchatFournisseur(): ?Achat
    {
        return $this->achatFournisseur;
    }

    public function setAchatFournisseur(?Achat $achatFournisseur): self
    {
        $this->achatFournisseur = $achatFournisseur;
        return $this;
    }

    /**
     * Calcule le prix de vente recommandé automatiquement
     */
    public function calculerPrixVenteRecommande(): void
    {
        if ($this->prixAchat && $this->margeMinimale) {
            $this->prixVenteRecommande = (string)((float)$this->prixAchat + (float)$this->margeMinimale);
        }
    }

    public function getDateEntree(): ?\DateTimeInterface
    {
        return $this->dateEntree;
    }

    public function setDateEntree(\DateTimeInterface $dateEntree): self
    {
        $this->dateEntree = $dateEntree;
        return $this;
    }

    public function getFournisseur(): ?Fournisseur
    {
        return $this->fournisseur;
    }

    public function setFournisseur(?Fournisseur $fournisseur): self
    {
        $this->fournisseur = $fournisseur;
        return $this;
    }

    public function getCaracteristiquesValeurs(): ?array
    {
        return $this->caracteristiquesValeurs ?? [];
    }

    public function setCaracteristiquesValeurs(?array $caracteristiquesValeurs): self
    {
        $this->caracteristiquesValeurs = $caracteristiquesValeurs;
        return $this;
    }

    public function getMarge(): float
    {
        return (float)$this->prixVente - (float)$this->prixAchat;
    }

    public function getMargePourcentage(): float
    {
        if ((float)$this->prixAchat == 0) {
            return 0;
        }
        return (($this->getMarge() / (float)$this->prixAchat) * 100);
    }

    public function isActif(): bool
    {
        return $this->actif;
    }

    public function setActif(bool $actif): self
    {
        $this->actif = $actif;
        return $this;
    }

    public function getSousCategorie(): ?SousCategorie
    {
        return $this->sousCategorie;
    }

    public function setSousCategorie(?SousCategorie $sousCategorie): self
    {
        $this->sousCategorie = $sousCategorie;
        return $this;
    }
}

