<?php

namespace App\Form;

use App\Entity\Caracteristique;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

class CaracteristiqueType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('nom', TextType::class, [
                'label' => 'Nom de la caractéristique',
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'Ex: Couleur, Stockage, État...'
                ]
            ])
            ->add('type', ChoiceType::class, [
                'label' => 'Type',
                'choices' => [
                    'Texte libre' => Caracteristique::TYPE_TEXT,
                    'Nombre' => Caracteristique::TYPE_NUMBER,
                    'Oui/Non' => Caracteristique::TYPE_BOOLEAN,
                    'Liste déroulante' => Caracteristique::TYPE_SELECT,
                    'Sélection multiple' => Caracteristique::TYPE_MULTI_SELECT,
                ],
                'attr' => [
                    'class' => 'form-select',
                    'data-caracteristique-type' => true
                ]
            ])
            ->add('obligatoire', CheckboxType::class, [
                'label' => 'Obligatoire lors de la création d\'un article',
                'required' => false,
                'attr' => [
                    'class' => 'form-check-input'
                ]
            ])
            ->add('valeursPossiblesText', TextareaType::class, [
                'label' => 'Valeurs possibles (séparées par des virgules)',
                'required' => false,
                'mapped' => false,
                'attr' => [
                    'class' => 'form-control valeurs-possibles-field',
                    'rows' => 3,
                    'placeholder' => 'Ex: Noir, Blanc, Rouge, Bleu'
                ]
            ]);

        // Gestion dynamique des valeurs possibles selon le type
        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
            $caracteristique = $event->getData();
            $form = $event->getForm();

            if ($caracteristique && $caracteristique->getId()) {
                // Pour une caractéristique existante, remplir le champ avec les valeurs
                $valeurs = $caracteristique->getValeursPossibles() ?? [];
                $valeursText = implode(', ', $valeurs);
                $form->get('valeursPossiblesText')->setData($valeursText);
            }
        });

        // Convertir les valeurs possibles AVANT la validation (PRE_SUBMIT)
        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
            $data = $event->getData();
            $form = $event->getForm();
            $caracteristique = $form->getData();

            if ($caracteristique && isset($data['type'])) {
                // Convertir les valeurs possibles en tableau si nécessaire
                $type = $data['type'];
                if (in_array($type, [Caracteristique::TYPE_SELECT, Caracteristique::TYPE_MULTI_SELECT])) {
                    $valeursText = $data['valeursPossiblesText'] ?? '';
                    if ($valeursText) {
                        $valeursArray = array_filter(array_map('trim', explode(',', $valeursText)));
                        $caracteristique->setValeursPossibles($valeursArray);
                    } else {
                        $caracteristique->setValeursPossibles([]);
                    }
                } else {
                    $caracteristique->setValeursPossibles(null);
                }
            }
        });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Caracteristique::class,
        ]);
    }
}

