<?php

namespace App\Form;

use App\Entity\Fournisseur;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

class FournisseurType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('nom', TextType::class, [
                'label' => 'Nom du fournisseur',
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'Ex: MarocPhone, SamsungPro...'
                ]
            ])
            ->add('type', ChoiceType::class, [
                'label' => 'Type',
                'choices' => [
                    'Wholesaler' => Fournisseur::TYPE_WHOLESALER,
                    'Distributeur' => Fournisseur::TYPE_DISTRIBUTEUR,
                    'Particulier' => Fournisseur::TYPE_PARTICULIER,
                    'Fabricant' => Fournisseur::TYPE_FABRICANT,
                    'Importateur' => Fournisseur::TYPE_IMPORTATEUR,
                ],
                'attr' => [
                    'class' => 'form-select'
                ]
            ])
            ->add('telephonesText', TextareaType::class, [
                'label' => 'Téléphones (un par ligne)',
                'required' => false,
                'mapped' => false,
                'attr' => [
                    'class' => 'form-control',
                    'rows' => 3,
                    'placeholder' => '0600123456' . "\n" . '0612345678'
                ],
                'help' => 'Entrez un numéro par ligne'
            ])
            ->add('email', EmailType::class, [
                'label' => 'Email',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'placeholder' => 'contact@fournisseur.com'
                ]
            ])
            ->add('adresse', TextareaType::class, [
                'label' => 'Adresse',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'rows' => 2
                ]
            ])
            ->add('ville', TextType::class, [
                'label' => 'Ville',
                'required' => false,
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('pays', TextType::class, [
                'label' => 'Pays',
                'required' => false,
                'attr' => [
                    'class' => 'form-control'
                ]
            ])
            ->add('notes', TextareaType::class, [
                'label' => 'Notes internes',
                'required' => false,
                'attr' => [
                    'class' => 'form-control',
                    'rows' => 4
                ]
            ]);

        // Gestion des téléphones multiples
        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
            $fournisseur = $event->getData();
            $form = $event->getForm();

            if ($fournisseur && $fournisseur->getId()) {
                $telephones = $fournisseur->getTelephones();
                $telephonesText = implode("\n", $telephones);
                $form->get('telephonesText')->setData($telephonesText);
            }
        });

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
            $data = $event->getData();
            $form = $event->getForm();
            $fournisseur = $form->getData();

            if (isset($data['telephonesText'])) {
                $telephonesText = $data['telephonesText'];
                $telephones = array_filter(array_map('trim', explode("\n", $telephonesText)));
                $fournisseur->setTelephones($telephones);
            }
        });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Fournisseur::class,
        ]);
    }
}
