<?php

namespace App\Service;

use App\Entity\Magasin;
use App\Repository\ProduitRepository;
use App\Repository\VenteRepository;
use App\Repository\AchatRepository;
use App\Repository\DepenseRepository;
use Doctrine\ORM\EntityManagerInterface;

class StatistiquesService
{
    public function __construct(
        private ProduitRepository $produitRepository,
        private VenteRepository $venteRepository,
        private AchatRepository $achatRepository,
        private DepenseRepository $depenseRepository,
        private EntityManagerInterface $em
    ) {
    }

    public function getStatistiquesMagasin(Magasin $magasin, ?\DateTime $dateDebut = null, ?\DateTime $dateFin = null): array
    {
        $dateDebut = $dateDebut ?? new \DateTime('first day of this month');
        $dateFin = $dateFin ?? new \DateTime('now');

        // Stock
        $produits = $this->produitRepository->findBy(['magasin' => $magasin, 'actif' => true]);
        $stockTotal = array_sum(array_map(fn($p) => $p->getQuantite(), $produits));
        $valeurStock = array_sum(array_map(fn($p) => $p->getQuantite() * (float)$p->getPrixAchat(), $produits));
        $stockFaible = count(array_filter($produits, fn($p) => $p->getQuantite() < 5));

        // Ventes
        $ventes = $this->venteRepository->findVentesByMagasinAndDate($magasin, $dateDebut, $dateFin);
        $totalVentes = array_sum(array_map(fn($v) => (float)$v->getMontantFinal(), $ventes));
        $margeTotale = array_sum(array_map(fn($v) => $v->getMargeTotale(), $ventes));

        // Achats
        $achats = $this->achatRepository->findAchatsByMagasinAndDate($magasin, $dateDebut, $dateFin);
        $totalAchats = array_sum(array_map(fn($a) => (float)$a->getMontantTotal(), $achats));

        // Dépenses
        $depenses = $this->depenseRepository->findDepensesByMagasinAndDate($magasin, $dateDebut, $dateFin);
        $totalDepenses = array_sum(array_map(fn($d) => (float)$d->getMontant(), $depenses));

        // Bénéfice
        $beneficeNet = $totalVentes - $totalAchats - $totalDepenses;

        // Top produits
        $topProduits = $this->getTopProduits($magasin, $dateDebut, $dateFin);

        // Ratio neuf/occasion
        // Note: etat (neuf/occasion) n'existe plus, on utilise maintenant statut
        // Pour l'instant, on ne filtre pas par état, mais on pourrait ajouter un champ caractéristique pour cela
        $neuf = 0; // À implémenter avec caractéristiques si nécessaire
        $occasion = 0; // À implémenter avec caractéristiques si nécessaire
        $ratioNeuf = $stockTotal > 0 ? ($neuf / $stockTotal) * 100 : 0;
        $ratioOccasion = $stockTotal > 0 ? ($occasion / $stockTotal) * 100 : 0;

        // Calcul du stock par sous-catégorie
        $stockParSousCategorie = $this->getStockParSousCategorie($magasin);

        return [
            'stock' => [
                'total' => $stockTotal,
                'faible' => $stockFaible,
                'valeur' => $valeurStock,
                'parSousCategorie' => $stockParSousCategorie,
            ],
            'ventes' => [
                'total' => $totalVentes,
                'marge' => $margeTotale,
                'nombre' => count($ventes),
            ],
            'achats' => [
                'total' => $totalAchats,
                'nombre' => count($achats),
            ],
            'depenses' => [
                'total' => $totalDepenses,
                'nombre' => count($depenses),
            ],
            'benefice' => $beneficeNet,
            'topProduits' => $topProduits,
            'ratio' => [
                'neuf' => $ratioNeuf,
                'occasion' => $ratioOccasion,
            ],
        ];
    }

    /**
     * Calcule le stock par sous-catégorie
     */
    public function getStockParSousCategorie(Magasin $magasin): array
    {
        $produits = $this->produitRepository->findBy(['magasin' => $magasin, 'actif' => true]);
        
        $stockParSousCategorie = [];
        
        foreach ($produits as $produit) {
            $sousCategorie = $produit->getSousCategorie();
            if (!$sousCategorie) {
                continue;
            }
            
            $sousCategorieId = $sousCategorie->getId();
            $sousCategorieNom = $sousCategorie->getNom();
            $categorieNom = $sousCategorie->getCategorie() ? $sousCategorie->getCategorie()->getNom() : 'Sans catégorie';
            
            if (!isset($stockParSousCategorie[$sousCategorieId])) {
                $stockParSousCategorie[$sousCategorieId] = [
                    'sousCategorie' => $sousCategorieNom,
                    'categorie' => $categorieNom,
                    'quantite' => 0,
                    'valeur' => 0.0,
                    'nombreProduits' => 0,
                ];
            }
            
            $stockParSousCategorie[$sousCategorieId]['quantite'] += $produit->getQuantite();
            $stockParSousCategorie[$sousCategorieId]['valeur'] += $produit->getQuantite() * (float)$produit->getPrixAchat();
            $stockParSousCategorie[$sousCategorieId]['nombreProduits']++;
        }
        
        // Trier par quantité décroissante
        uasort($stockParSousCategorie, fn($a, $b) => $b['quantite'] <=> $a['quantite']);
        
        return $stockParSousCategorie;
    }

    private function getTopProduits(Magasin $magasin, \DateTime $dateDebut, \DateTime $dateFin, int $limit = 10): array
    {
        $qb = $this->em->createQueryBuilder();
        $qb->select('p.id, p.nom, SUM(lv.quantite) as quantite_vendue, SUM(lv.montantTotal) as total_vendu')
            ->from('App\Entity\LigneVente', 'lv')
            ->join('lv.vente', 'v')
            ->join('lv.produit', 'p')
            ->where('v.magasin = :magasin')
            ->andWhere('v.dateVente BETWEEN :dateDebut AND :dateFin')
            ->andWhere('v.retour = false')
            ->setParameter('magasin', $magasin)
            ->setParameter('dateDebut', $dateDebut)
            ->setParameter('dateFin', $dateFin)
            ->groupBy('p.id, p.nom')
            ->orderBy('total_vendu', 'DESC')
            ->setMaxResults($limit);

        return $qb->getQuery()->getResult();
    }

    public function getVentesParPeriode(Magasin $magasin, string $periode = 'jour'): array
    {
        $qb = $this->em->createQueryBuilder();
        
        // Récupérer les ventes avec leurs dates et montants
        $qb->select('v.dateVente, v.montantFinal')
            ->from('App\Entity\Vente', 'v')
            ->where('v.magasin = :magasin')
            ->andWhere('v.retour = false')
            ->setParameter('magasin', $magasin)
            ->orderBy('v.dateVente', 'ASC');

        $ventes = $qb->getQuery()->getResult();
        
        // Grouper et formater en PHP (compatible avec toutes les bases de données)
        $grouped = [];
        foreach ($ventes as $vente) {
            $date = $vente['dateVente'];
            if ($date instanceof \DateTime) {
                $key = $periode === 'jour' 
                    ? $date->format('Y-m-d') 
                    : $date->format('Y-m');
            } else {
                $dateObj = new \DateTime($date);
                $key = $periode === 'jour' 
                    ? $dateObj->format('Y-m-d') 
                    : $dateObj->format('Y-m');
            }
            
            if (!isset($grouped[$key])) {
                $grouped[$key] = 0;
            }
            $grouped[$key] += (float)$vente['montantFinal'];
        }
        
        // Convertir en format attendu
        $result = [];
        foreach ($grouped as $periodeKey => $total) {
            $result[] = [
                'periode' => $periodeKey,
                'total' => $total
            ];
        }
        
        return $result;
    }
}

