<?php

namespace App\Validator\Constraints;

use App\Entity\Caracteristique;
use App\Repository\CaracteristiqueRepository;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

class CaracteristiqueValideValidator extends ConstraintValidator
{
    public function __construct(
        private CaracteristiqueRepository $caracteristiqueRepository
    ) {
    }

    public function validate($value, Constraint $constraint): void
    {
        if (!$constraint instanceof CaracteristiqueValide) {
            throw new UnexpectedTypeException($constraint, CaracteristiqueValide::class);
        }

        if (!$value instanceof Caracteristique) {
            return;
        }

        // Vérifier que la caractéristique appartient soit à une catégorie soit à une sous-catégorie (jamais les deux)
        $hasCategorie = $value->getCategorie() !== null;
        $hasSousCategorie = $value->getSousCategorie() !== null;

        if ($hasCategorie && $hasSousCategorie) {
            $this->context->buildViolation($constraint->messageCategorieSousCategorie)
                ->addViolation();
            return;
        }

        if (!$hasCategorie && !$hasSousCategorie) {
            $this->context->buildViolation($constraint->messageCategorieSousCategorie)
                ->addViolation();
            return;
        }

        // Vérifier l'unicité du nom dans la catégorie ou sous-catégorie
        $existing = null;
        if ($hasCategorie) {
            $existing = $this->caracteristiqueRepository->findOneByNomAndCategorie(
                $value->getNom(),
                $value->getCategorie(),
                $value->getId()
            );
        } elseif ($hasSousCategorie) {
            $existing = $this->caracteristiqueRepository->findOneByNomAndSousCategorie(
                $value->getNom(),
                $value->getSousCategorie(),
                $value->getId()
            );
        }

        if ($existing) {
            $this->context->buildViolation('Une caractéristique avec ce nom existe déjà dans cette catégorie/sous-catégorie.')
                ->atPath('nom')
                ->addViolation();
        }

        // Vérifier que si type = select ou multi-select, alors valeurs_possibles est obligatoire
        if ($value->requiresValeursPossibles()) {
            $valeurs = $value->getValeursPossibles();
            if (empty($valeurs) || count($valeurs) === 0) {
                $this->context->buildViolation($constraint->messageValeursPossibles)
                    ->atPath('valeursPossibles')
                    ->addViolation();
            }
        }
    }
}

