<?php

namespace App\Controller;

use App\Repository\VenteRepository;
use App\Repository\AchatRepository;
use App\Repository\DepenseRepository;
use App\Repository\ClientRepository;
use App\Repository\FournisseurRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/bilan', name: 'bilan_')]
#[IsGranted('ROLE_USER')]
class BilanController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(
        Request $request,
        VenteRepository $venteRepository,
        AchatRepository $achatRepository,
        DepenseRepository $depenseRepository,
        ClientRepository $clientRepository,
        FournisseurRepository $fournisseurRepository,
        EntityManagerInterface $em
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        // Récupérer toutes les ventes, achats et dépenses du magasin
        $ventes = $venteRepository->createQueryBuilder('v')
            ->where('v.magasin = :magasin')
            ->andWhere('v.retour = false')
            ->setParameter('magasin', $magasin)
            ->orderBy('v.dateVente', 'ASC')
            ->getQuery()
            ->getResult();

        $achats = $achatRepository->createQueryBuilder('a')
            ->where('a.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->orderBy('a.dateAchat', 'ASC')
            ->getQuery()
            ->getResult();

        $depenses = $depenseRepository->createQueryBuilder('d')
            ->where('d.magasin = :magasin')
            ->setParameter('magasin', $magasin)
            ->orderBy('d.dateDepense', 'ASC')
            ->getQuery()
            ->getResult();

        // Calculer les crédits clients
        $clients = $clientRepository->findByMagasin($magasin, '', true);
        $totalCreditsClients = 0;
        foreach ($clients as $client) {
            foreach ($client->getVentes() as $vente) {
                if (!$vente->isRetour() && (float)$vente->getMontantRestant() > 0) {
                    $totalCreditsClients += (float)$vente->getMontantRestant();
                }
            }
        }

        // Calculer les dettes fournisseurs
        $fournisseurs = $fournisseurRepository->findBy(['magasin' => $magasin]);
        $totalDettesFournisseurs = 0;
        foreach ($fournisseurs as $fournisseur) {
            foreach ($fournisseur->getAchats() as $achat) {
                $montantPaye = $achat->getMontantPaye();
                $montantTotal = (float)$achat->getMontantTotal();
                $montantRestant = max(0, $montantTotal - $montantPaye);
                $totalDettesFournisseurs += $montantRestant;
            }
        }

        // Organiser les données par mois
        $bilansParMois = [];
        $bilanTotal = [
            'totalVentes' => 0,
            'totalAchats' => 0,
            'totalDepenses' => 0,
            'totalCreditsClients' => $totalCreditsClients,
            'totalDettesFournisseurs' => $totalDettesFournisseurs,
            'profitBrut' => 0,
            'profitNet' => 0,
        ];

        // Grouper par mois
        $moisData = [];
        
        // Ventes par mois
        foreach ($ventes as $vente) {
            $moisKey = $vente->getDateVente()->format('Y-m');
            if (!isset($moisData[$moisKey])) {
                $moisLabel = $vente->getDateVente()->format('m/Y');
                $moisData[$moisKey] = [
                    'mois' => $vente->getDateVente()->format('Y-m'),
                    'moisLabel' => $moisLabel,
                    'ventes' => [],
                    'achats' => [],
                    'depenses' => [],
                    'totalVentes' => 0,
                    'totalAchats' => 0,
                    'totalDepenses' => 0,
                ];
            }
            $moisData[$moisKey]['ventes'][] = $vente;
            $moisData[$moisKey]['totalVentes'] += (float)$vente->getMontantFinal();
            $bilanTotal['totalVentes'] += (float)$vente->getMontantFinal();
        }

        // Achats par mois
        foreach ($achats as $achat) {
            $moisKey = $achat->getDateAchat()->format('Y-m');
            if (!isset($moisData[$moisKey])) {
                $moisLabel = $achat->getDateAchat()->format('m/Y');
                $moisData[$moisKey] = [
                    'mois' => $achat->getDateAchat()->format('Y-m'),
                    'moisLabel' => $moisLabel,
                    'ventes' => [],
                    'achats' => [],
                    'depenses' => [],
                    'totalVentes' => 0,
                    'totalAchats' => 0,
                    'totalDepenses' => 0,
                ];
            }
            $moisData[$moisKey]['achats'][] = $achat;
            $moisData[$moisKey]['totalAchats'] += (float)$achat->getMontantTotal();
            $bilanTotal['totalAchats'] += (float)$achat->getMontantTotal();
        }

        // Dépenses par mois
        foreach ($depenses as $depense) {
            $moisKey = $depense->getDateDepense()->format('Y-m');
            if (!isset($moisData[$moisKey])) {
                $moisLabel = $depense->getDateDepense()->format('m/Y');
                $moisData[$moisKey] = [
                    'mois' => $depense->getDateDepense()->format('Y-m'),
                    'moisLabel' => $moisLabel,
                    'ventes' => [],
                    'achats' => [],
                    'depenses' => [],
                    'totalVentes' => 0,
                    'totalAchats' => 0,
                    'totalDepenses' => 0,
                ];
            }
            $moisData[$moisKey]['depenses'][] = $depense;
            $moisData[$moisKey]['totalDepenses'] += (float)$depense->getMontant();
            $bilanTotal['totalDepenses'] += (float)$depense->getMontant();
        }

        // Calculer les profits par mois et total
        foreach ($moisData as $moisKey => $data) {
            $data['profitBrut'] = $data['totalVentes'] - $data['totalAchats'];
            $data['profitNet'] = $data['profitBrut'] - $data['totalDepenses'];
            $bilansParMois[] = $data;
        }

        // Trier par mois (plus récent en premier)
        usort($bilansParMois, function($a, $b) {
            return strcmp($b['mois'], $a['mois']);
        });

        // Calculer le bilan total
        $bilanTotal['profitBrut'] = $bilanTotal['totalVentes'] - $bilanTotal['totalAchats'];
        $bilanTotal['profitNet'] = $bilanTotal['profitBrut'] - $bilanTotal['totalDepenses'];

        return $this->render('bilan/index.html.twig', [
            'bilansParMois' => $bilansParMois,
            'bilanTotal' => $bilanTotal,
        ]);
    }
}

