<?php

namespace App\Controller;

use App\Entity\Caracteristique;
use App\Entity\Categorie;
use App\Entity\SousCategorie;
use App\Form\CaracteristiqueType;
use App\Repository\CaracteristiqueRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/caracteristiques', name: 'caracteristiques_')]
#[IsGranted('ROLE_MANAGER')]
class CaracteristiqueController extends AbstractController
{
    #[Route('/categorie/{categorieId}/new', name: 'new_categorie', methods: ['GET', 'POST'])]
    public function newForCategorie(int $categorieId, Request $request, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $categorie = $em->getRepository(Categorie::class)->findOneBy([
            'id' => $categorieId,
            'magasin' => $magasin
        ]);

        if (!$categorie) {
            throw $this->createNotFoundException('Catégorie non trouvée');
        }

        $caracteristique = new Caracteristique();
        $caracteristique->setCategorie($categorie);
        $caracteristique->setMagasin($magasin);

        $form = $this->createForm(CaracteristiqueType::class, $caracteristique);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->persist($caracteristique);
            $em->flush();

            $logService->log($user, $magasin, 'Création caractéristique', 'Caracteristique', $caracteristique->getId(), "Caractéristique: {$caracteristique->getNom()}");

            return $this->redirectToRoute('categories_show', ['id' => $categorieId]);
        }

        return $this->render('caracteristique/new.html.twig', [
            'caracteristique' => $caracteristique,
            'categorie' => $categorie,
            'form' => $form,
        ]);
    }

    #[Route('/sous-categorie/{sousCategorieId}/new', name: 'new_sous_categorie', methods: ['GET', 'POST'])]
    public function newForSousCategorie(int $sousCategorieId, Request $request, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $sousCategorie = $em->getRepository(SousCategorie::class)->findOneBy([
            'id' => $sousCategorieId,
            'magasin' => $magasin
        ]);

        if (!$sousCategorie) {
            throw $this->createNotFoundException('Sous-catégorie non trouvée');
        }

        $caracteristique = new Caracteristique();
        $caracteristique->setSousCategorie($sousCategorie);
        $caracteristique->setMagasin($magasin);

        $form = $this->createForm(CaracteristiqueType::class, $caracteristique);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->persist($caracteristique);
            $em->flush();

            $logService->log($user, $magasin, 'Création caractéristique', 'Caracteristique', $caracteristique->getId(), "Caractéristique: {$caracteristique->getNom()}");

            return $this->redirectToRoute('categories_show', ['id' => $sousCategorie->getCategorie()->getId()]);
        }

        return $this->render('caracteristique/new.html.twig', [
            'caracteristique' => $caracteristique,
            'sousCategorie' => $sousCategorie,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Caracteristique $caracteristique, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($caracteristique->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(CaracteristiqueType::class, $caracteristique);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->flush();

            $logService->log($user, $magasin, 'Modification caractéristique', 'Caracteristique', $caracteristique->getId(), "Caractéristique: {$caracteristique->getNom()}");

            $redirectId = $caracteristique->getCategorie() 
                ? $caracteristique->getCategorie()->getId() 
                : $caracteristique->getSousCategorie()->getCategorie()->getId();

            return $this->redirectToRoute('categories_show', ['id' => $redirectId]);
        }

        return $this->render('caracteristique/edit.html.twig', [
            'caracteristique' => $caracteristique,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/delete', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, Caracteristique $caracteristique, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($caracteristique->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $caracteristique->getId(), $request->request->get('_token'))) {
            $nom = $caracteristique->getNom();
            $categorieId = $caracteristique->getCategorie() 
                ? $caracteristique->getCategorie()->getId() 
                : $caracteristique->getSousCategorie()->getCategorie()->getId();
            
            $em->remove($caracteristique);
            $em->flush();

            $logService->log($user, $magasin, 'Suppression caractéristique', 'Caracteristique', null, "Caractéristique: {$nom}");

            $this->addFlash('success', 'Caractéristique supprimée avec succès');

            return $this->redirectToRoute('categories_show', ['id' => $categorieId]);
        }

        return $this->redirectToRoute('categories_index');
    }
}

