<?php

namespace App\Controller;

use App\Entity\Categorie;
use App\Form\CategorieType;
use App\Repository\CategorieRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/categories', name: 'categories_')]
#[IsGranted('ROLE_MANAGER')]
class CategorieController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(CategorieRepository $categorieRepository): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $categories = $categorieRepository->findBy(['magasin' => $magasin]);

        return $this->render('categorie/index.html.twig', [
            'categories' => $categories,
        ]);
    }

    #[Route('/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $categorie = new Categorie();
        $categorie->setMagasin($magasin);

        $form = $this->createForm(CategorieType::class, $categorie);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->persist($categorie);
            $em->flush();

            $logService->log($user, $magasin, 'Création catégorie', 'Categorie', $categorie->getId(), "Catégorie: {$categorie->getNom()}");

            return $this->redirectToRoute('categories_index');
        }

        return $this->render('categorie/new.html.twig', [
            'categorie' => $categorie,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'show', methods: ['GET'])]
    public function show(Categorie $categorie): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($categorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        return $this->render('categorie/show.html.twig', [
            'categorie' => $categorie,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Categorie $categorie, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($categorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(CategorieType::class, $categorie);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->flush();

            $logService->log($user, $magasin, 'Modification catégorie', 'Categorie', $categorie->getId(), "Catégorie: {$categorie->getNom()}");

            return $this->redirectToRoute('categories_show', ['id' => $categorie->getId()]);
        }

        return $this->render('categorie/edit.html.twig', [
            'categorie' => $categorie,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/delete', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, Categorie $categorie, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($categorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $categorie->getId(), $request->request->get('_token'))) {
            $nom = $categorie->getNom();
            
            $em->remove($categorie);
            $em->flush();

            $logService->log($user, $magasin, 'Suppression catégorie', 'Categorie', null, "Catégorie: {$nom}");

            $this->addFlash('success', 'Catégorie supprimée avec succès');
        }

        return $this->redirectToRoute('categories_index');
    }
}

