<?php

namespace App\Controller;

use App\Entity\Client;
use App\Entity\Fournisseur;
use App\Entity\PaiementClient;
use App\Entity\Paiement;
use App\Entity\Vente;
use App\Entity\Achat;
use App\Repository\ClientRepository;
use App\Repository\FournisseurRepository;
use App\Repository\VenteRepository;
use App\Repository\AchatRepository;
use App\Repository\ProduitRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/credits', name: 'credits_')]
#[IsGranted('ROLE_USER')]
class CreditController extends AbstractController
{
    #[Route('', name: 'index', methods: ['GET'])]
    public function index(
        ClientRepository $clientRepository,
        FournisseurRepository $fournisseurRepository,
        Request $request,
        EntityManagerInterface $em
    ): Response {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $type = $request->query->get('type', 'clients'); // 'clients' ou 'fournisseurs'
        $search = $request->query->get('search', '');

        // Récupérer les clients avec crédit
        $clientsAvecCredit = [];
        if ($type === 'clients' || $type === 'all') {
            $clients = $clientRepository->findByMagasin($magasin, $search, true);
            foreach ($clients as $client) {
                $totalRestant = 0;
                $ventesImpayees = [];
                foreach ($client->getVentes() as $vente) {
                    // Vérifier que la vente appartient bien au magasin
                    if ($vente->getMagasin() !== $magasin) {
                        continue;
                    }
                    // Recalculer le montant restant basé sur les paiements réels
                    $montantPaye = (float)$vente->getMontantPaye();
                    $montantFinal = (float)$vente->getMontantFinal();
                    $montantRestant = max(0, $montantFinal - $montantPaye);
                    
                    // Mettre à jour le montant restant dans la base si nécessaire
                    if (abs((float)$vente->getMontantRestant() - $montantRestant) > 0.01) {
                        $vente->setMontantRestant((string)$montantRestant);
                        $vente->mettreAJourStatutPaiement();
                        $em->persist($vente);
                    }
                    
                    if (!$vente->isRetour() && $montantRestant > 0) {
                        $totalRestant += $montantRestant;
                        $ventesImpayees[] = $vente;
                    }
                }
                if ($totalRestant > 0) {
                    $clientsAvecCredit[] = [
                        'client' => $client,
                        'totalRestant' => $totalRestant,
                        'ventesImpayees' => $ventesImpayees,
                    ];
                }
            }
            $em->flush(); // Sauvegarder les mises à jour de montantRestant
        }

        // Récupérer les fournisseurs avec dette
        $fournisseursAvecDette = [];
        if ($type === 'fournisseurs' || $type === 'all') {
            $fournisseurs = $fournisseurRepository->findBy(['magasin' => $magasin], ['nom' => 'ASC']);
            foreach ($fournisseurs as $fournisseur) {
                $totalDette = 0;
                $achatsImpayes = [];
                foreach ($fournisseur->getAchats() as $achat) {
                    // Vérifier que l'achat appartient bien au magasin
                    if ($achat->getMagasin() !== $magasin) {
                        continue;
                    }
                    
                    // Recalculer le montant restant basé sur les paiements réels
                    $montantPaye = $achat->getMontantPaye();
                    $montantTotal = (float)$achat->getMontantTotal();
                    $montantRestant = max(0, $montantTotal - $montantPaye);
                    
                    // Mettre à jour le montant restant dans la base si nécessaire
                    if (abs((float)$achat->getMontantRestant() - $montantRestant) > 0.01) {
                        $achat->setMontantRestant((string)$montantRestant);
                        $em->persist($achat);
                    }
                    
                    if ($montantRestant > 0) {
                        $totalDette += $montantRestant;
                        $achatsImpayes[] = $achat;
                    }
                }
                if ($totalDette > 0) {
                    $fournisseursAvecDette[] = [
                        'fournisseur' => $fournisseur,
                        'totalDette' => $totalDette,
                        'achatsImpayes' => $achatsImpayes,
                    ];
                }
            }
            $em->flush(); // Sauvegarder les mises à jour de montantRestant
        }

        // Calculer les totaux
        $totalCreditsClients = array_sum(array_column($clientsAvecCredit, 'totalRestant'));
        $totalDettesFournisseurs = array_sum(array_column($fournisseursAvecDette, 'totalDette'));

        return $this->render('credit/index.html.twig', [
            'clientsAvecCredit' => $clientsAvecCredit,
            'fournisseursAvecDette' => $fournisseursAvecDette,
            'totalCreditsClients' => $totalCreditsClients,
            'totalDettesFournisseurs' => $totalDettesFournisseurs,
            'type' => $type,
            'search' => $search,
        ]);
    }

    #[Route('/client/{id}/paiement', name: 'client_paiement', methods: ['POST'])]
    public function ajouterPaiementClient(
        Client $client,
        Request $request,
        EntityManagerInterface $em,
        LogService $logService
    ): JsonResponse {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($client->getMagasin() !== $magasin) {
            return new JsonResponse(['error' => 'Accès refusé'], 403);
        }

        $data = json_decode($request->getContent(), true);
        $venteId = $data['vente_id'] ?? null;
        $montant = (float)($data['montant'] ?? 0);
        $mode = $data['mode'] ?? PaiementClient::MODE_CASH;
        $reference = $data['reference'] ?? null;
        $notes = $data['notes'] ?? null;
        $datePaiement = isset($data['date']) ? new \DateTime($data['date']) : new \DateTime();

        if ($montant <= 0) {
            return new JsonResponse(['error' => 'Le montant doit être supérieur à 0'], 400);
        }

        $vente = $em->getRepository(Vente::class)->find($venteId);
        if (!$vente || $vente->getClient() !== $client || $vente->getMagasin() !== $magasin) {
            return new JsonResponse(['error' => 'Vente non trouvée'], 404);
        }

        // Vérifier que le montant ne dépasse pas le reste dû
        $montantRestant = (float)$vente->getMontantRestant();
        if ($montant > $montantRestant) {
            return new JsonResponse(['error' => 'Le montant dépasse le reste dû'], 400);
        }

        $paiement = new PaiementClient();
        $paiement->setClient($client);
        $paiement->setVente($vente);
        $paiement->setUtilisateur($user);
        $paiement->setMontant((string)$montant);
        $paiement->setMode($mode);
        $paiement->setReference($reference);
        $paiement->setNotes($notes);
        $paiement->setDatePaiement($datePaiement);

        $em->persist($paiement);

        // Ajouter le paiement à la collection de la vente pour établir la relation bidirectionnelle
        $vente->addPaiementClient($paiement);

        // Recalculer automatiquement le montant payé et le statut à partir de tous les paiements
        $vente->mettreAJourStatutPaiement();

        $em->flush();

        $logService->log(
            $user,
            $magasin,
            'Paiement client',
            'PaiementClient',
            $paiement->getId(),
            "Paiement de {$montant} MAD pour la vente #{$vente->getNumeroFacture()}"
        );

        return new JsonResponse([
            'success' => true,
            'paiement' => [
                'id' => $paiement->getId(),
                'montant' => $paiement->getMontant(),
                'date' => $paiement->getDatePaiement()->format('Y-m-d H:i:s'),
            ],
            'vente' => [
                'montantPaye' => $vente->getMontantPaye(),
                'montantRestant' => $vente->getMontantRestant(),
                'statutPaiement' => $vente->getStatutPaiement(),
            ],
        ]);
    }

    #[Route('/fournisseur/{id}/paiement', name: 'fournisseur_paiement', methods: ['POST'])]
    public function ajouterPaiementFournisseur(
        Fournisseur $fournisseur,
        Request $request,
        EntityManagerInterface $em,
        LogService $logService
    ): JsonResponse {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($fournisseur->getMagasin() !== $magasin) {
            return new JsonResponse(['error' => 'Accès refusé'], 403);
        }

        $data = json_decode($request->getContent(), true);
        $achatId = $data['achat_id'] ?? null;
        $montant = (float)($data['montant'] ?? 0);
        $mode = $data['mode'] ?? Paiement::MODE_ESPECES;
        $reference = $data['reference'] ?? null;
        $notes = $data['notes'] ?? null;
        $datePaiement = isset($data['date']) ? new \DateTime($data['date']) : new \DateTime();

        if ($montant <= 0) {
            return new JsonResponse(['error' => 'Le montant doit être supérieur à 0'], 400);
        }

        $achat = $em->getRepository(Achat::class)->find($achatId);
        if (!$achat || $achat->getFournisseur() !== $fournisseur || $achat->getMagasin() !== $magasin) {
            return new JsonResponse(['error' => 'Achat non trouvé'], 404);
        }

        // Vérifier que le montant ne dépasse pas le reste dû
        $montantRestant = (float)$achat->getMontantRestant();
        if ($montant > $montantRestant) {
            return new JsonResponse(['error' => 'Le montant dépasse le reste dû'], 400);
        }

        $paiement = new Paiement();
        $paiement->setFournisseur($fournisseur);
        $paiement->setAchat($achat);
        $paiement->setUtilisateur($user);
        $paiement->setMontant((string)$montant);
        $paiement->setMode($mode);
        $paiement->setReference($reference);
        $paiement->setNotes($notes);
        $paiement->setDatePaiement($datePaiement);

        $em->persist($paiement);

        // Mettre à jour l'achat
        $nouveauMontantRestant = max(0, $montantRestant - $montant);
        $achat->setMontantRestant((string)$nouveauMontantRestant);

        $em->flush();

        $logService->log(
            $user,
            $magasin,
            'Paiement fournisseur',
            'Paiement',
            $paiement->getId(),
            "Paiement de {$montant} MAD pour l'achat #{$achat->getNumeroFacture()}"
        );

        return new JsonResponse([
            'success' => true,
            'paiement' => [
                'id' => $paiement->getId(),
                'montant' => $paiement->getMontant(),
                'date' => $paiement->getDatePaiement()->format('Y-m-d H:i:s'),
            ],
            'achat' => [
                'montantRestant' => $achat->getMontantRestant(),
            ],
        ]);
    }
}

