<?php

namespace App\Controller;

use App\Entity\SousCategorie;
use App\Entity\Categorie;
use App\Entity\Caracteristique;
use App\Form\SousCategorieType;
use App\Repository\SousCategorieRepository;
use App\Service\LogService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/sous-categories', name: 'sous_categories_')]
#[IsGranted('ROLE_MANAGER')]
class SousCategorieController extends AbstractController
{
    #[Route('/categorie/{categorieId}', name: 'index', methods: ['GET'])]
    public function index(int $categorieId, SousCategorieRepository $sousCategorieRepository, EntityManagerInterface $em): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $sousCategories = $sousCategorieRepository->findByCategorie($categorieId);

        // Vérifier que la catégorie appartient au magasin
        $categorie = $em->getRepository(Categorie::class)
            ->findOneBy(['id' => $categorieId, 'magasin' => $magasin]);

        if (!$categorie) {
            throw $this->createNotFoundException('Catégorie non trouvée');
        }

        return $this->render('sous_categorie/index.html.twig', [
            'sousCategories' => $sousCategories,
            'categorie' => $categorie,
        ]);
    }

    #[Route('/categorie/{categorieId}/new', name: 'new', methods: ['GET', 'POST'])]
    public function new(int $categorieId, Request $request, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        $categorie = $em->getRepository(Categorie::class)->findOneBy([
            'id' => $categorieId,
            'magasin' => $magasin
        ]);

        if (!$categorie) {
            throw $this->createNotFoundException('Catégorie non trouvée');
        }

        $sousCategorie = new SousCategorie();
        $sousCategorie->setCategorie($categorie);
        $sousCategorie->setMagasin($magasin);

        $form = $this->createForm(SousCategorieType::class, $sousCategorie);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->persist($sousCategorie);
            $em->flush();

            $logService->log($user, $magasin, 'Création sous-catégorie', 'SousCategorie', $sousCategorie->getId(), "Sous-catégorie: {$sousCategorie->getNom()}");

            return $this->redirectToRoute('categories_show', ['id' => $categorieId]);
        }

        return $this->render('sous_categorie/new.html.twig', [
            'sousCategorie' => $sousCategorie,
            'categorie' => $categorie,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, SousCategorie $sousCategorie, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($sousCategorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        $form = $this->createForm(SousCategorieType::class, $sousCategorie);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->flush();

            $logService->log($user, $magasin, 'Modification sous-catégorie', 'SousCategorie', $sousCategorie->getId(), "Sous-catégorie: {$sousCategorie->getNom()}");

            return $this->redirectToRoute('categories_show', ['id' => $sousCategorie->getCategorie()->getId()]);
        }

        return $this->render('sous_categorie/edit.html.twig', [
            'sousCategorie' => $sousCategorie,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/duplicate', name: 'duplicate', methods: ['GET', 'POST'])]
    public function duplicate(SousCategorie $sousCategorie, EntityManagerInterface $em, LogService $logService, SousCategorieRepository $sousCategorieRepository): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($sousCategorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        // Générer un nom unique pour la copie
        $nomBase = $sousCategorie->getNom();
        $nouveauNom = $nomBase . ' - Copie';
        $categorie = $sousCategorie->getCategorie();
        
        // Vérifier si le nom existe déjà et générer un nom unique
        $compteur = 1;
        while (true) {
            $existing = $sousCategorieRepository->findOneBy([
                'nom' => $nouveauNom,
                'categorie' => $categorie
            ]);
            
            if (!$existing) {
                break; // Le nom est disponible
            }
            
            $compteur++;
            $nouveauNom = $nomBase . ' - Copie ' . $compteur;
        }

        // Créer une nouvelle sous-catégorie en copiant l'existante
        $nouvelleSousCategorie = new SousCategorie();
        $nouvelleSousCategorie->setNom($nouveauNom);
        $nouvelleSousCategorie->setDescription($sousCategorie->getDescription());
        $nouvelleSousCategorie->setCategorie($categorie);
        $nouvelleSousCategorie->setMagasin($magasin);

        $em->persist($nouvelleSousCategorie);
        $em->flush();

        // Dupliquer les caractéristiques
        foreach ($sousCategorie->getCaracteristiques() as $caracteristique) {
            $nouvelleCaracteristique = new Caracteristique();
            $nouvelleCaracteristique->setNom($caracteristique->getNom());
            $nouvelleCaracteristique->setType($caracteristique->getType());
            $nouvelleCaracteristique->setValeursPossibles($caracteristique->getValeursPossibles());
            $nouvelleCaracteristique->setObligatoire($caracteristique->isObligatoire());
            $nouvelleCaracteristique->setSousCategorie($nouvelleSousCategorie);
            $nouvelleCaracteristique->setMagasin($magasin);

            $em->persist($nouvelleCaracteristique);
        }

        $em->flush();

        $logService->log($user, $magasin, 'Duplication sous-catégorie', 'SousCategorie', $nouvelleSousCategorie->getId(), "Sous-catégorie: {$nouvelleSousCategorie->getNom()} (copie de: {$sousCategorie->getNom()})");

        $this->addFlash('success', 'Sous-catégorie dupliquée avec succès');

        return $this->redirectToRoute('categories_show', ['id' => $sousCategorie->getCategorie()->getId()]);
    }

    #[Route('/{id}/delete', name: 'delete', methods: ['POST'])]
    public function delete(Request $request, SousCategorie $sousCategorie, EntityManagerInterface $em, LogService $logService): Response
    {
        $user = $this->getUser();
        $magasin = $user->getMagasin();

        if ($sousCategorie->getMagasin() !== $magasin) {
            throw $this->createAccessDeniedException();
        }

        if ($this->isCsrfTokenValid('delete' . $sousCategorie->getId(), $request->request->get('_token'))) {
            $categorieId = $sousCategorie->getCategorie()->getId();
            $nom = $sousCategorie->getNom();
            
            $em->remove($sousCategorie);
            $em->flush();

            $logService->log($user, $magasin, 'Suppression sous-catégorie', 'SousCategorie', null, "Sous-catégorie: {$nom}");

            $this->addFlash('success', 'Sous-catégorie supprimée avec succès');
        }

        return $this->redirectToRoute('categories_show', ['id' => $categorieId ?? null]);
    }
}

