<?php

namespace App\Entity;

use App\Repository\FournisseurRepository;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity(repositoryClass: FournisseurRepository::class)]
#[ORM\Table(name: 'fournisseurs')]
#[ORM\UniqueConstraint(name: 'unique_nom_magasin', columns: ['nom', 'magasin_id'])]
class Fournisseur
{
    const TYPE_WHOLESALER = 'wholesaler';
    const TYPE_DISTRIBUTEUR = 'distributeur';
    const TYPE_PARTICULIER = 'particulier';
    const TYPE_FABRICANT = 'fabricant';
    const TYPE_IMPORTATEUR = 'importateur';

    const STATUT_ACTIF = 'actif';
    const STATUT_DESACTIVE = 'desactive';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column]
    private ?int $id = null;

    #[ORM\Column(length: 255)]
    private ?string $nom = null;

    #[ORM\Column(length: 50)]
    private ?string $type = self::TYPE_DISTRIBUTEUR;

    #[ORM\Column(type: Types::JSON, nullable: true)]
    private ?array $telephones = []; // Support multiple téléphones

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $email = null;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $adresse = null;

    #[ORM\Column(length: 100, nullable: true)]
    private ?string $ville = null;

    #[ORM\Column(length: 100, nullable: true)]
    private ?string $pays = null;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $notes = null;

    #[ORM\Column(length: 20)]
    private ?string $statut = self::STATUT_ACTIF;

    #[ORM\ManyToOne(inversedBy: 'fournisseurs')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Magasin $magasin = null;

    #[ORM\OneToMany(mappedBy: 'fournisseur', targetEntity: Produit::class)]
    private Collection $produits;

    #[ORM\OneToMany(mappedBy: 'fournisseur', targetEntity: Achat::class)]
    private Collection $achats;

    #[ORM\OneToMany(mappedBy: 'fournisseur', targetEntity: Paiement::class)]
    private Collection $paiements;

    #[ORM\Column(type: Types::DATETIME_MUTABLE)]
    private ?\DateTimeInterface $dateCreation = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
    private ?\DateTimeInterface $dateModification = null;

    public function __construct()
    {
        $this->produits = new ArrayCollection();
        $this->achats = new ArrayCollection();
        $this->paiements = new ArrayCollection();
        $this->telephones = [];
        $this->dateCreation = new \DateTime();
        $this->statut = self::STATUT_ACTIF;
        $this->type = self::TYPE_DISTRIBUTEUR;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getNom(): ?string
    {
        return $this->nom;
    }

    public function setNom(string $nom): self
    {
        $this->nom = $nom;
        return $this;
    }

    public function getType(): ?string
    {
        return $this->type;
    }

    public function setType(string $type): self
    {
        $this->type = $type;
        return $this;
    }

    public function getTelephones(): ?array
    {
        return $this->telephones ?? [];
    }

    public function setTelephones(?array $telephones): self
    {
        $this->telephones = $telephones ?? [];
        return $this;
    }

    public function getTelephone(): ?string
    {
        // Pour compatibilité avec l'ancien code, retourner le premier téléphone
        $telephones = $this->getTelephones();
        return !empty($telephones) ? $telephones[0] : null;
    }

    public function setTelephone(?string $telephone): self
    {
        // Pour compatibilité avec l'ancien code
        if ($telephone) {
            $this->telephones = [$telephone];
        }
        return $this;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(?string $email): self
    {
        $this->email = $email;
        return $this;
    }

    public function getVille(): ?string
    {
        return $this->ville;
    }

    public function setVille(?string $ville): self
    {
        $this->ville = $ville;
        return $this;
    }

    public function getPays(): ?string
    {
        return $this->pays;
    }

    public function setPays(?string $pays): self
    {
        $this->pays = $pays;
        return $this;
    }

    public function getNotes(): ?string
    {
        return $this->notes;
    }

    public function setNotes(?string $notes): self
    {
        $this->notes = $notes;
        return $this;
    }

    public function getStatut(): ?string
    {
        return $this->statut;
    }

    public function setStatut(string $statut): self
    {
        $this->statut = $statut;
        return $this;
    }

    public function isActif(): bool
    {
        return $this->statut === self::STATUT_ACTIF;
    }

    public function getAdresse(): ?string
    {
        return $this->adresse;
    }

    public function setAdresse(?string $adresse): self
    {
        $this->adresse = $adresse;
        return $this;
    }

    public function getMagasin(): ?Magasin
    {
        return $this->magasin;
    }

    public function setMagasin(?Magasin $magasin): self
    {
        $this->magasin = $magasin;
        return $this;
    }

    /**
     * @return Collection<int, Produit>
     */
    public function getProduits(): Collection
    {
        return $this->produits;
    }

    /**
     * @return Collection<int, Achat>
     */
    public function getAchats(): Collection
    {
        return $this->achats;
    }

    public function getDateCreation(): ?\DateTimeInterface
    {
        return $this->dateCreation;
    }

    public function setDateCreation(\DateTimeInterface $dateCreation): self
    {
        $this->dateCreation = $dateCreation;
        return $this;
    }

    public function getDateModification(): ?\DateTimeInterface
    {
        return $this->dateModification;
    }

    public function setDateModification(?\DateTimeInterface $dateModification): self
    {
        $this->dateModification = $dateModification;
        return $this;
    }

    /**
     * @return Collection<int, Paiement>
     */
    public function getPaiements(): Collection
    {
        return $this->paiements;
    }

    public function addPaiement(Paiement $paiement): self
    {
        if (!$this->paiements->contains($paiement)) {
            $this->paiements->add($paiement);
            $paiement->setFournisseur($this);
        }
        return $this;
    }

    public function removePaiement(Paiement $paiement): self
    {
        if ($this->paiements->removeElement($paiement)) {
            if ($paiement->getFournisseur() === $this) {
                $paiement->setFournisseur(null);
            }
        }
        return $this;
    }

    /**
     * Calcule le total des achats
     */
    public function getTotalAchete(): float
    {
        $total = 0;
        foreach ($this->achats as $achat) {
            $total += (float)$achat->getMontantTotal();
        }
        return $total;
    }

    /**
     * Calcule le total payé
     */
    public function getTotalPaye(): float
    {
        $total = 0;
        foreach ($this->paiements as $paiement) {
            $total += (float)$paiement->getMontant();
        }
        return $total;
    }

    /**
     * Calcule le reste dû
     */
    public function getResteDu(): float
    {
        return $this->getTotalAchete() - $this->getTotalPaye();
    }
}

