<?php

namespace App\Service;

use App\Repository\ProduitRepository;
use Doctrine\ORM\EntityManagerInterface;

class CodeBarresService
{
    public function __construct(
        private ProduitRepository $produitRepository,
        private EntityManagerInterface $em
    ) {
    }

    /**
     * Génère un code-barres unique
     * @param array $codesBarresExclus Liste de codes-barres à exclure (déjà utilisés dans la session courante)
     */
    public function genererCodeBarres(array $codesBarresExclus = []): string
    {
        $maxAttempts = 100;
        $attempt = 0;
        
        do {
            // Génère un code-barres unique de 13 chiffres (format EAN-13)
            $prefix = '200'; // Préfixe pour identifier nos produits
            $random = str_pad((string)mt_rand(0, 999999999), 9, '0', STR_PAD_LEFT);
            $code = $prefix . $random;
            
            // Calcul du chiffre de contrôle (checksum EAN-13)
            $checksum = $this->calculerChecksumEAN13($code);
            $codeBarres = $code . $checksum;
            
            // Vérifier si le code est dans la liste d'exclusion
            if (in_array($codeBarres, $codesBarresExclus, true)) {
                $attempt++;
                continue;
            }
            
            // Vérifier l'unicité dans la base de données
            $existant = $this->produitRepository->findOneBy(['codeBarres' => $codeBarres]);
            $attempt++;
            
            if (!$existant) {
                return $codeBarres;
            }
        } while ($attempt < $maxAttempts);
        
        // Si on n'a pas trouvé de code unique après 100 tentatives, utiliser un timestamp + microtime
        $microtime = substr((string)microtime(true), -9);
        $code = $prefix . str_pad($microtime, 9, '0', STR_PAD_LEFT);
        $checksum = $this->calculerChecksumEAN13($code);
        $codeBarres = $code . $checksum;
        
        // Vérifier une dernière fois l'unicité
        if (in_array($codeBarres, $codesBarresExclus, true)) {
            // Si toujours en conflit, ajouter un random supplémentaire
            $codeBarres = $codeBarres . mt_rand(0, 9);
        }
        
        return $codeBarres;
    }

    private function calculerChecksumEAN13(string $code): string
    {
        $sum = 0;
        for ($i = 0; $i < 12; $i++) {
            $digit = (int)$code[$i];
            $sum += ($i % 2 === 0) ? $digit : $digit * 3;
        }
        $checksum = (10 - ($sum % 10)) % 10;
        return (string)$checksum;
    }

    public function validerCodeBarres(string $code): bool
    {
        if (strlen($code) !== 13 || !ctype_digit($code)) {
            return false;
        }
        
        $codeSansChecksum = substr($code, 0, 12);
        $checksumAttendu = $this->calculerChecksumEAN13($codeSansChecksum);
        $checksumReel = $code[12];
        
        return $checksumAttendu === $checksumReel;
    }

    /**
     * Génère une référence interne unique basée sur le code-barres
     * Format: REF-{code-barres} ou REF-{année}-{numéro séquentiel}
     */
    public function genererReferenceInterne(?string $codeBarres = null): string
    {
        if ($codeBarres) {
            // Utiliser le code-barres comme base de la référence
            return 'REF-' . $codeBarres;
        }
        
        // Sinon, générer une référence basée sur l'année et un numéro séquentiel
        $annee = date('Y');
        $maxAttempts = 100;
        $attempt = 0;
        
        do {
            $numero = str_pad((string)mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);
            $reference = 'REF-' . $annee . '-' . $numero;
            
            // Vérifier l'unicité
            $existant = $this->produitRepository->findOneBy(['referenceInterne' => $reference]);
            $attempt++;
            
            if (!$existant) {
                return $reference;
            }
        } while ($attempt < $maxAttempts);
        
        // Si on n'a pas trouvé de référence unique, utiliser un timestamp
        $timestamp = substr((string)time(), -8);
        return 'REF-' . $annee . '-' . $timestamp;
    }
}

