<?php

namespace App\Service;

use App\Repository\VenteRepository;
use App\Repository\AchatRepository;
use Doctrine\ORM\EntityManagerInterface;

class FactureService
{
    public function __construct(
        private VenteRepository $venteRepository,
        private AchatRepository $achatRepository,
        private EntityManagerInterface $em
    ) {
    }

    /**
     * Génère un numéro de facture unique pour les ventes
     * Format: FAC-YYYYMMDD-XXXXX (ex: FAC-20251203-00001)
     */
    public function genererNumeroFacture(): string
    {
        $date = date('Ymd');
        $prefix = 'FAC-' . $date . '-';
        
        // Récupérer le dernier numéro de facture du jour (ventes et achats)
        $derniereVente = $this->venteRepository->createQueryBuilder('v')
            ->where('v.numeroFacture LIKE :prefix')
            ->setParameter('prefix', $prefix . '%')
            ->orderBy('v.numeroFacture', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
        
        $dernierAchat = $this->achatRepository->createQueryBuilder('a')
            ->where('a.numeroFacture LIKE :prefix')
            ->setParameter('prefix', $prefix . '%')
            ->orderBy('a.numeroFacture', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
        
        $dernierNumero = 0;
        if ($derniereVente) {
            $numero = (int)substr($derniereVente->getNumeroFacture(), strlen($prefix));
            $dernierNumero = max($dernierNumero, $numero);
        }
        if ($dernierAchat) {
            $numero = (int)substr($dernierAchat->getNumeroFacture(), strlen($prefix));
            $dernierNumero = max($dernierNumero, $numero);
        }
        
        $dernierNumero++;
        
        return $prefix . str_pad((string)$dernierNumero, 5, '0', STR_PAD_LEFT);
    }
    
    /**
     * Génère un numéro de facture unique pour les achats
     * Format: ACH-YYYYMMDD-XXXXX (ex: ACH-20251203-00001)
     */
    public function genererNumeroFactureAchat(): string
    {
        $date = date('Ymd');
        $prefix = 'ACH-' . $date . '-';
        
        // Récupérer le dernier numéro de facture d'achat du jour
        $dernierAchat = $this->achatRepository->createQueryBuilder('a')
            ->where('a.numeroFacture LIKE :prefix')
            ->setParameter('prefix', $prefix . '%')
            ->orderBy('a.numeroFacture', 'DESC')
            ->setMaxResults(1)
            ->getQuery()
            ->getOneOrNullResult();
        
        if ($dernierAchat) {
            // Extraire le numéro séquentiel
            $numero = (int)substr($dernierAchat->getNumeroFacture(), strlen($prefix));
            $numero++;
        } else {
            $numero = 1;
        }
        
        return $prefix . str_pad((string)$numero, 5, '0', STR_PAD_LEFT);
    }
}

